/*
 * bsc1225429_net_ethtool_privflags
 *
 * Fix for CVE-2021-47517, bsc#1225429
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/ethtool/netlink.h */
#include <linux/ethtool_netlink.h>
#include <linux/netdevice.h>
#include <net/genetlink.h>
#include <net/sock.h>

struct ethnl_req_info {
	struct net_device	*dev;
	u32			flags;
};

struct ethnl_reply_data {
	struct net_device		*dev;
};

void ethnl_ops_complete(struct net_device *dev);
int ethnl_ops_begin(struct net_device *dev);
int klpp_ethnl_parse_header_dev_get(struct ethnl_req_info *req_info,
                                    const struct nlattr *header, struct net *net,
                                    struct netlink_ext_ack *extack, bool require_dev);

/* klp-ccp: from net/ethtool/common.h */
#include <linux/netdevice.h>
#include <linux/ethtool.h>

/* klp-ccp: from net/ethtool/bitset.h */
typedef const char (*const ethnl_string_array_t)[ETH_GSTRING_LEN];

static int (*klpe_ethnl_bitset_is_compact)(const struct nlattr *bitset, bool *compact);

static int (*klpe_ethnl_update_bitset32)(u32 *bitmap, unsigned int nbits,
			  const struct nlattr *attr, ethnl_string_array_t names,
			  struct netlink_ext_ack *extack, bool *mod);

/* klp-ccp: from net/ethtool/privflags.c */
struct privflags_reply_data {
	struct ethnl_reply_data		base;
	const char			(*priv_flag_names)[ETH_GSTRING_LEN];
	unsigned int			n_priv_flags;
	u32				priv_flags;
};

#define PRIVFLAGS_REPDATA(__reply_base) \
	container_of(__reply_base, struct privflags_reply_data, base)

static int (*klpe_ethnl_get_priv_flags_info)(struct net_device *dev,
				     unsigned int *count,
				     const char (**names)[ETH_GSTRING_LEN]);

int klpp_privflags_prepare_data(const struct ethnl_req_info *req_base,
				  struct ethnl_reply_data *reply_base,
				  struct genl_info *info)
{
	struct privflags_reply_data *data = PRIVFLAGS_REPDATA(reply_base);
	struct net_device *dev = reply_base->dev;
	const char (*names)[ETH_GSTRING_LEN];
	const struct ethtool_ops *ops;
	unsigned int nflags;
	int ret;

	ops = dev->ethtool_ops;
	if (!ops->get_priv_flags || !ops->get_sset_count || !ops->get_strings)
		return -EOPNOTSUPP;
	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		return ret;

	ret = (*klpe_ethnl_get_priv_flags_info)(dev, &nflags, &names);
	if (ret < 0)
		goto out_ops;
	data->priv_flags = ops->get_priv_flags(dev);
	data->priv_flag_names = names;
	data->n_priv_flags = nflags;

out_ops:
	ethnl_ops_complete(dev);
	return ret;
}

int klpp_ethnl_set_privflags(struct sk_buff *skb, struct genl_info *info)
{
	const char (*names)[ETH_GSTRING_LEN] = NULL;
	struct ethnl_req_info req_info = {};
	struct nlattr **tb = info->attrs;
	const struct ethtool_ops *ops;
	struct net_device *dev;
	unsigned int nflags;
	bool mod = false;
	bool compact;
	u32 flags;
	int ret;

	if (!tb[ETHTOOL_A_PRIVFLAGS_FLAGS])
		return -EINVAL;
	ret = (*klpe_ethnl_bitset_is_compact)(tb[ETHTOOL_A_PRIVFLAGS_FLAGS], &compact);
	if (ret < 0)
		return ret;
	ret = klpp_ethnl_parse_header_dev_get(&req_info,
					 tb[ETHTOOL_A_PRIVFLAGS_HEADER],
					 genl_info_net(info), info->extack,
					 true);
	if (ret < 0)
		return ret;
	dev = req_info.dev;
	ops = dev->ethtool_ops;
	ret = -EOPNOTSUPP;
	if (!ops->get_priv_flags || !ops->set_priv_flags ||
	    !ops->get_sset_count || !ops->get_strings)
		goto out_dev;

	rtnl_lock();
	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		goto out_rtnl;
	ret = (*klpe_ethnl_get_priv_flags_info)(dev, &nflags, compact ? NULL : &names);
	if (ret < 0)
		goto out_ops;
	flags = ops->get_priv_flags(dev);

	ret = (*klpe_ethnl_update_bitset32)(&flags, nflags,
				    tb[ETHTOOL_A_PRIVFLAGS_FLAGS], names,
				    info->extack, &mod);
	if (ret < 0 || !mod)
		goto out_free;
	ret = ops->set_priv_flags(dev, flags);
	if (ret < 0)
		goto out_free;
	ethtool_notify(dev, ETHTOOL_MSG_PRIVFLAGS_NTF, NULL);

out_free:
	kfree(names);
out_ops:
	ethnl_ops_complete(dev);
out_rtnl:
	rtnl_unlock();
out_dev:
	dev_put(dev);
	return ret;
}

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "ethnl_bitset_is_compact", (void *)&klpe_ethnl_bitset_is_compact },
	{ "ethnl_get_priv_flags_info",
	  (void *)&klpe_ethnl_get_priv_flags_info },
	{ "ethnl_update_bitset32", (void *)&klpe_ethnl_update_bitset32 },
};

int bsc1225429_net_ethtool_privflags_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

