/*
 * bsc1225429_net_ethtool_fec
 *
 * Fix for CVE-2021-47517, bsc#1225429
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/ethtool/netlink.h */
#include <linux/ethtool_netlink.h>
#include <linux/netdevice.h>
#include <net/genetlink.h>
#include <net/sock.h>

static inline void ethnl_update_u8(u8 *dst, const struct nlattr *attr,
				   bool *mod)
{
	u8 val;

	if (!attr)
		return;
	val = nla_get_u8(attr);
	if (*dst == val)
		return;

	*dst = val;
	*mod = true;
}

struct ethnl_req_info {
	struct net_device	*dev;
	u32			flags;
};

struct ethnl_reply_data {
	struct net_device		*dev;
};

void ethnl_ops_complete(struct net_device *dev);
int ethnl_ops_begin(struct net_device *dev);
int klpp_ethnl_parse_header_dev_get(struct ethnl_req_info *req_info,
                                    const struct nlattr *header, struct net *net,
                                    struct netlink_ext_ack *extack, bool require_dev);

/* klp-ccp: from net/ethtool/common.h */
#include <linux/ethtool.h>

static const char (*klpe_link_mode_names)[][ETH_GSTRING_LEN];

/* klp-ccp: from net/ethtool/bitset.h */
typedef const char (*const ethnl_string_array_t)[ETH_GSTRING_LEN];

static int (*klpe_ethnl_update_bitset)(unsigned long *bitmap, unsigned int nbits,
			const struct nlattr *attr, ethnl_string_array_t names,
			struct netlink_ext_ack *extack, bool *mod);

/* klp-ccp: from net/ethtool/fec.c */
struct fec_reply_data {
	struct ethnl_reply_data		base;
	__ETHTOOL_DECLARE_LINK_MODE_MASK(fec_link_modes);
	u32 active_fec;
	u8 fec_auto;
	struct fec_stat_grp {
		u64 stats[1 + ETHTOOL_MAX_LANES];
		u8 cnt;
	} corr, uncorr, corr_bits;
};

#define FEC_REPDATA(__reply_base) \
	container_of(__reply_base, struct fec_reply_data, base)

static void
(*klpe_ethtool_fec_to_link_modes)(u32 fec, unsigned long *link_modes, u8 *fec_auto);

static int
ethtool_link_modes_to_fecparam(struct ethtool_fecparam *fec,
			       unsigned long *link_modes, u8 fec_auto)
{
	memset(fec, 0, sizeof(*fec));

	if (fec_auto)
		fec->fec |= ETHTOOL_FEC_AUTO;

	if (__test_and_clear_bit(ETHTOOL_LINK_MODE_FEC_NONE_BIT, link_modes))
		fec->fec |= ETHTOOL_FEC_OFF;
	if (__test_and_clear_bit(ETHTOOL_LINK_MODE_FEC_RS_BIT, link_modes))
		fec->fec |= ETHTOOL_FEC_RS;
	if (__test_and_clear_bit(ETHTOOL_LINK_MODE_FEC_BASER_BIT, link_modes))
		fec->fec |= ETHTOOL_FEC_BASER;
	if (__test_and_clear_bit(ETHTOOL_LINK_MODE_FEC_LLRS_BIT, link_modes))
		fec->fec |= ETHTOOL_FEC_LLRS;

	if (!bitmap_empty(link_modes, __ETHTOOL_LINK_MODE_MASK_NBITS))
		return -EINVAL;

	return 0;
}

static void
(*klpe_fec_stats_recalc)(struct fec_stat_grp *grp, struct ethtool_fec_stat *stats);

int klpp_fec_prepare_data(const struct ethnl_req_info *req_base,
			    struct ethnl_reply_data *reply_base,
			    struct genl_info *info)
{
	__ETHTOOL_DECLARE_LINK_MODE_MASK(active_fec_modes) = {};
	struct fec_reply_data *data = FEC_REPDATA(reply_base);
	struct net_device *dev = reply_base->dev;
	struct ethtool_fecparam fec = {};
	int ret;

	if (!dev->ethtool_ops->get_fecparam)
		return -EOPNOTSUPP;
	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		return ret;
	ret = dev->ethtool_ops->get_fecparam(dev, &fec);
	if (ret)
		goto out_complete;
	if (req_base->flags & ETHTOOL_FLAG_STATS &&
	    dev->ethtool_ops->get_fec_stats) {
		struct ethtool_fec_stats stats;

		ethtool_stats_init((u64 *)&stats, sizeof(stats) / 8);
		dev->ethtool_ops->get_fec_stats(dev, &stats);

		(*klpe_fec_stats_recalc)(&data->corr, &stats.corrected_blocks);
		(*klpe_fec_stats_recalc)(&data->uncorr, &stats.uncorrectable_blocks);
		(*klpe_fec_stats_recalc)(&data->corr_bits, &stats.corrected_bits);
	}

	WARN_ON_ONCE(fec.reserved);

	(*klpe_ethtool_fec_to_link_modes)(fec.fec, data->fec_link_modes,
				  &data->fec_auto);

	(*klpe_ethtool_fec_to_link_modes)(fec.active_fec, active_fec_modes, NULL);
	data->active_fec = find_first_bit(active_fec_modes,
					  __ETHTOOL_LINK_MODE_MASK_NBITS);
	/* Don't report attr if no FEC mode set. Note that
	 * ethtool_fecparam_to_link_modes() ignores NONE and AUTO.
	 */
	if (data->active_fec == __ETHTOOL_LINK_MODE_MASK_NBITS)
		data->active_fec = 0;

out_complete:
	ethnl_ops_complete(dev);
	return ret;
}

int klpp_ethnl_set_fec(struct sk_buff *skb, struct genl_info *info)
{
	__ETHTOOL_DECLARE_LINK_MODE_MASK(fec_link_modes) = {};
	struct ethnl_req_info req_info = {};
	struct nlattr **tb = info->attrs;
	struct ethtool_fecparam fec = {};
	const struct ethtool_ops *ops;
	struct net_device *dev;
	bool mod = false;
	u8 fec_auto;
	int ret;

	ret = klpp_ethnl_parse_header_dev_get(&req_info, tb[ETHTOOL_A_FEC_HEADER],
					 genl_info_net(info), info->extack,
					 true);
	if (ret < 0)
		return ret;
	dev = req_info.dev;
	ops = dev->ethtool_ops;
	ret = -EOPNOTSUPP;
	if (!ops->get_fecparam || !ops->set_fecparam)
		goto out_dev;

	rtnl_lock();
	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		goto out_rtnl;
	ret = ops->get_fecparam(dev, &fec);
	if (ret < 0)
		goto out_ops;

	(*klpe_ethtool_fec_to_link_modes)(fec.fec, fec_link_modes, &fec_auto);

	ret = (*klpe_ethnl_update_bitset)(fec_link_modes,
				  __ETHTOOL_LINK_MODE_MASK_NBITS,
				  tb[ETHTOOL_A_FEC_MODES],
				  (*klpe_link_mode_names), info->extack, &mod);
	if (ret < 0)
		goto out_ops;
	ethnl_update_u8(&fec_auto, tb[ETHTOOL_A_FEC_AUTO], &mod);

	ret = 0;
	if (!mod)
		goto out_ops;

	ret = ethtool_link_modes_to_fecparam(&fec, fec_link_modes, fec_auto);
	if (ret) {
		NL_SET_ERR_MSG_ATTR(info->extack, tb[ETHTOOL_A_FEC_MODES],
				    "invalid FEC modes requested");
		goto out_ops;
	}
	if (!fec.fec) {
		ret = -EINVAL;
		NL_SET_ERR_MSG_ATTR(info->extack, tb[ETHTOOL_A_FEC_MODES],
				    "no FEC modes set");
		goto out_ops;
	}

	ret = dev->ethtool_ops->set_fecparam(dev, &fec);
	if (ret < 0)
		goto out_ops;
	ethtool_notify(dev, ETHTOOL_MSG_FEC_NTF, NULL);

out_ops:
	ethnl_ops_complete(dev);
out_rtnl:
	rtnl_unlock();
out_dev:
	dev_put(dev);
	return ret;
}

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "ethtool_fec_to_link_modes",
	  (void *)&klpe_ethtool_fec_to_link_modes },
	{ "fec_stats_recalc", (void *)&klpe_fec_stats_recalc },
	{ "link_mode_names", (void *)&klpe_link_mode_names },
};

int bsc1225429_net_ethtool_fec_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

