/*
 * bsc1225429_net_ethtool_cabletest
 *
 * Fix for CVE-2021-47517, bsc#1225429
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from net/ethtool/cabletest.c */
#include <linux/phy.h>
/* klp-ccp: from net/ethtool/netlink.h */
#include <linux/ethtool_netlink.h>
#include <linux/netdevice.h>
#include <net/genetlink.h>
#include <net/sock.h>

struct ethnl_req_info {
	struct net_device	*dev;
	u32			flags;
};

void ethnl_ops_complete(struct net_device *dev);
int ethnl_ops_begin(struct net_device *dev);
int klpp_ethnl_parse_header_dev_get(struct ethnl_req_info *req_info,
                                    const struct nlattr *header, struct net *net,
                                    struct netlink_ext_ack *extack, bool require_dev);

/* klp-ccp: from net/ethtool/common.h */
#include <linux/netdevice.h>
#include <linux/ethtool.h>

static const struct ethtool_phy_ops *(*klpe_ethtool_phy_ops);

/* klp-ccp: from net/ethtool/cabletest.c */
#define MAX_CABLE_LENGTH_CM (150 * 100)

static int (*klpe_ethnl_cable_test_started)(struct phy_device *phydev, u8 cmd);

int klpp_ethnl_act_cable_test(struct sk_buff *skb, struct genl_info *info)
{
	struct ethnl_req_info req_info = {};
	const struct ethtool_phy_ops *ops;
	struct nlattr **tb = info->attrs;
	struct net_device *dev;
	int ret;

	ret = klpp_ethnl_parse_header_dev_get(&req_info,
					 tb[ETHTOOL_A_CABLE_TEST_HEADER],
					 genl_info_net(info), info->extack,
					 true);
	if (ret < 0)
		return ret;

	dev = req_info.dev;
	if (!dev->phydev) {
		ret = -EOPNOTSUPP;
		goto out_dev_put;
	}

	rtnl_lock();
	ops = (*klpe_ethtool_phy_ops);
	if (!ops || !ops->start_cable_test) {
		ret = -EOPNOTSUPP;
		goto out_rtnl;
	}

	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		goto out_rtnl;

	ret = ops->start_cable_test(dev->phydev, info->extack);

	ethnl_ops_complete(dev);

	if (!ret)
		(*klpe_ethnl_cable_test_started)(dev->phydev,
					 ETHTOOL_MSG_CABLE_TEST_NTF);

out_rtnl:
	rtnl_unlock();
out_dev_put:
	dev_put(dev);
	return ret;
}

static const struct nla_policy (*klpe_cable_test_tdr_act_cfg_policy)[5];

static int klpr_ethnl_act_cable_test_tdr_cfg(const struct nlattr *nest,
					struct genl_info *info,
					struct phy_tdr_config *cfg)
{
	struct nlattr *tb[ARRAY_SIZE((*klpe_cable_test_tdr_act_cfg_policy))];
	int ret;

	cfg->first = 100;
	cfg->step = 100;
	cfg->last = MAX_CABLE_LENGTH_CM;
	cfg->pair = PHY_PAIR_ALL;

	if (!nest)
		return 0;

	ret = nla_parse_nested(tb,
			       ARRAY_SIZE((*klpe_cable_test_tdr_act_cfg_policy)) - 1,
			       nest, (*klpe_cable_test_tdr_act_cfg_policy),
			       info->extack);
	if (ret < 0)
		return ret;

	if (tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_FIRST])
		cfg->first = nla_get_u32(
			tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_FIRST]);

	if (tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_LAST])
		cfg->last = nla_get_u32(tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_LAST]);

	if (tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_STEP])
		cfg->step = nla_get_u32(tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_STEP]);

	if (tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_PAIR]) {
		cfg->pair = nla_get_u8(tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_PAIR]);
		if (cfg->pair > ETHTOOL_A_CABLE_PAIR_D) {
			NL_SET_ERR_MSG_ATTR(
				info->extack,
				tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_PAIR],
				"invalid pair parameter");
			return -EINVAL;
		}
	}

	if (cfg->first > MAX_CABLE_LENGTH_CM) {
		NL_SET_ERR_MSG_ATTR(info->extack,
				    tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_FIRST],
				    "invalid first parameter");
		return -EINVAL;
	}

	if (cfg->last > MAX_CABLE_LENGTH_CM) {
		NL_SET_ERR_MSG_ATTR(info->extack,
				    tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_LAST],
				    "invalid last parameter");
		return -EINVAL;
	}

	if (cfg->first > cfg->last) {
		NL_SET_ERR_MSG(info->extack, "invalid first/last parameter");
		return -EINVAL;
	}

	if (!cfg->step) {
		NL_SET_ERR_MSG_ATTR(info->extack,
				    tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_STEP],
				    "invalid step parameter");
		return -EINVAL;
	}

	if (cfg->step > (cfg->last - cfg->first)) {
		NL_SET_ERR_MSG_ATTR(info->extack,
				    tb[ETHTOOL_A_CABLE_TEST_TDR_CFG_STEP],
				    "step parameter too big");
		return -EINVAL;
	}

	return 0;
}

int klpp_ethnl_act_cable_test_tdr(struct sk_buff *skb, struct genl_info *info)
{
	struct ethnl_req_info req_info = {};
	const struct ethtool_phy_ops *ops;
	struct nlattr **tb = info->attrs;
	struct phy_tdr_config cfg;
	struct net_device *dev;
	int ret;

	ret = klpp_ethnl_parse_header_dev_get(&req_info,
					 tb[ETHTOOL_A_CABLE_TEST_TDR_HEADER],
					 genl_info_net(info), info->extack,
					 true);
	if (ret < 0)
		return ret;

	dev = req_info.dev;
	if (!dev->phydev) {
		ret = -EOPNOTSUPP;
		goto out_dev_put;
	}

	ret = klpr_ethnl_act_cable_test_tdr_cfg(tb[ETHTOOL_A_CABLE_TEST_TDR_CFG],
					   info, &cfg);
	if (ret)
		goto out_dev_put;

	rtnl_lock();
	ops = (*klpe_ethtool_phy_ops);
	if (!ops || !ops->start_cable_test_tdr) {
		ret = -EOPNOTSUPP;
		goto out_rtnl;
	}

	ret = ethnl_ops_begin(dev);
	if (ret < 0)
		goto out_rtnl;

	ret = ops->start_cable_test_tdr(dev->phydev, info->extack, &cfg);

	ethnl_ops_complete(dev);

	if (!ret)
		(*klpe_ethnl_cable_test_started)(dev->phydev,
					 ETHTOOL_MSG_CABLE_TEST_TDR_NTF);

out_rtnl:
	rtnl_unlock();
out_dev_put:
	dev_put(dev);
	return ret;
}

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cable_test_tdr_act_cfg_policy",
	  (void *)&klpe_cable_test_tdr_act_cfg_policy },
	{ "ethnl_cable_test_started", (void *)&klpe_ethnl_cable_test_started },
	{ "ethtool_phy_ops", (void *)&klpe_ethtool_phy_ops },
};

int bsc1225429_net_ethtool_cabletest_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

