/*
 * livepatch_bsc1227753
 *
 * Fix for CVE-2023-52885, bsc#1227753
 *
 *  Upstream commit:
 *  fc80fc2d4e39 ("SUNRPC: Fix UAF in svc_tcp_listen_data_ready()")
 *
 *  SLE12-SP5 commit:
 *  78e70c4ab450ee2ad398e094ef2bc328320806fb
 *
 *  SLE15-SP3 commit:
 *  e97b8a4b218f5adfb34b51003ba8913c04a8c27f
 *
 *  SLE15-SP4 and -SP5 commit:
 *  28f43b1541bc27cadd3ccaa0a504000f093a1e3a
 *
 *  SLE15-SP6 commit:
 *  28f43b1541bc27cadd3ccaa0a504000f093a1e3a
 *
 *  SLE MICRO-6-0 commit:
 *  28f43b1541bc27cadd3ccaa0a504000f093a1e3a
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Fernando Gonzalez <fernando.gonzalez@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from net/sunrpc/svcsock.c */
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <linux/errno.h>

/* klp-ccp: from include/linux/fcntl.h */
#define _LINUX_FCNTL_H

/* klp-ccp: from include/uapi/asm-generic/fcntl.h */
#define O_WRONLY	00000001

#define O_APPEND	00002000

#define O_NONBLOCK	00004000

#define O_DSYNC		00010000	/* used to be O_SYNC, see below */

#define O_DIRECT	00040000	/* direct disk access hint */

#define O_NOATIME	01000000

#define __O_SYNC	04000000

#ifndef HAVE_ARCH_STRUCT_FLOCK

struct flock;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

/* klp-ccp: from include/uapi/linux/fcntl.h */
#define RWH_WRITE_LIFE_NONE	1
#define RWH_WRITE_LIFE_SHORT	2
#define RWH_WRITE_LIFE_MEDIUM	3
#define RWH_WRITE_LIFE_LONG	4
#define RWH_WRITE_LIFE_EXTREME	5

#define AT_FDCWD		-100    /* Special value used to indicate
                                           openat should use the current
                                           working directory. */
#define AT_SYMLINK_NOFOLLOW	0x100   /* Do not follow symbolic links.  */

#define AT_NO_AUTOMOUNT		0x800	/* Suppress terminal automount traversal */

/* klp-ccp: from net/sunrpc/svcsock.c */
#include <linux/net.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/udp.h>

#include <linux/unistd.h>
#include <linux/slab.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/file.h>

#include <net/sock.h>
#include <net/checksum.h>

/* klp-ccp: from include/net/ipv6.h */
#define _NET_IPV6_H

/* klp-ccp: from net/sunrpc/svcsock.c */
#include <net/ipv6.h>

#include <net/tcp_states.h>
#include <linux/uaccess.h>
#include <asm/ioctls.h>

/* klp-ccp: from include/linux/sunrpc/types.h */
#define _LINUX_SUNRPC_TYPES_H_

/* klp-ccp: from include/linux/sunrpc/debug.h */
#if IS_ENABLED(CONFIG_SUNRPC_DEBUG)
static unsigned int		(*klpe_rpc_debug);

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

/* klp-ccp: from net/sunrpc/svcsock.c */
#include <linux/sunrpc/clnt.h>
#include <linux/sunrpc/xdr.h>
#include <linux/sunrpc/msg_prot.h>
#include <linux/sunrpc/svcsock.h>
#include <linux/sunrpc/debug.h>

/* klp-ccp: from include/linux/sunrpc/svc_xprt.h */
static void	(*klpe_svc_xprt_enqueue)(struct svc_xprt *xprt);

/* klp-ccp: from net/sunrpc/svcsock.c */
#include <linux/sunrpc/stats.h>
#include <linux/sunrpc/xprt.h>

/* klp-ccp: from net/sunrpc/sunrpc.h */
#include <linux/net.h>

#define RPCDBG_FACILITY		RPCDBG_SVCXPRT

# define klpr_ifdebug(fac)		if (unlikely(*klpe_rpc_debug & RPCDBG_##fac))

# define klpr_dfprintk(fac, fmt, ...)				\
do {								\
	klpr_ifdebug(fac)					\
		printk(KERN_DEFAULT fmt, ##__VA_ARGS__);	\
} while (0)

#define klpr_dprintk(fmt, ...) \
	klpr_dfprintk(FACILITY, fmt, ##__VA_ARGS__)

/* klp-ccp: from net/sunrpc/svcsock.c */
void klpp_svc_tcp_listen_data_ready(struct sock *sk)
{
	struct svc_sock *svsk = (struct svc_sock *)sk->sk_user_data;

	klpr_dprintk("svc: socket %p TCP (listen) state change %d\n",
		     sk, sk->sk_state);

	/*
	 * This callback may called twice when a new connection
	 * is established as a child socket inherits everything
	 * from a parent LISTEN socket.
	 * 1) data_ready method of the parent socket will be called
	 *    when one of child sockets become ESTABLISHED.
	 * 2) data_ready method of the child socket may be called
	 *    when it receives data before the socket is accepted.
	 * In case of 2, we should ignore it silently and DO NOT
	 * dereference svsk.
	 */
	if (sk->sk_state != TCP_LISTEN)
		return;

	if (svsk) {
		/* Refer to svc_setup_socket() for details. */
		rmb();
		svsk->sk_odata(sk);
		set_bit(XPT_CONN, &svsk->sk_xprt.xpt_flags);
		(*klpe_svc_xprt_enqueue)(&svsk->sk_xprt);
	} else
		printk("svc: socket %p: no user data\n", sk);
}


#include "livepatch_bsc1227753.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "net/sunrpc/sunrpc"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "rpc_debug", (void *)&klpe_rpc_debug, "sunrpc" },
	{ "svc_xprt_enqueue", (void *)&klpe_svc_xprt_enqueue, "sunrpc" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1227753_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LP_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1227753_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
