/*
 * livepatch_bsc1229644
 *
 * Fix for CVE-2022-48912, bsc#1229644
 *
 *  Upstream commit:
 *  56763f12b0f0 ("netfilter: fix use-after-free in __nf_register_net_hook()")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP2 and -SP3 commit:
 *  c4dd672e1110f115948ee3802d73f18e0d803d03
 *
 *  SLE15-SP4 and -SP5 commit:
 *  f8f42c3ca843bf8cc51fb197d10e460f52a96afe
 *
 *  SLE15-SP6 commit:
 *  Not affected
 *
 *  SLE MICRO-6-0 commit:
 *  Not affected
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo MEZZELA <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/netfilter/core.c */
#include <linux/netfilter.h>
#include <linux/netdevice.h>

#include <linux/mutex.h>
#include <net/net_namespace.h>

#include <net/sock.h>

/* klp-ccp: from net/netfilter/nf_internals.h */
#include <linux/netdevice.h>

/* klp-ccp: from net/netfilter/core.c */
#ifdef CONFIG_JUMP_LABEL
extern struct static_key nf_hooks_needed[NFPROTO_NUMPROTO][NF_MAX_HOOKS];

extern typeof(nf_hooks_needed) nf_hooks_needed;

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

static struct mutex (*klpe_nf_hook_mutex);

static void (*klpe___nf_hook_entries_free)(struct rcu_head *h);

static void klpr_nf_hook_entries_free(struct nf_hook_entries *e)
{
	struct nf_hook_entries_rcu_head *head;
	struct nf_hook_ops **ops;
	unsigned int num;

	if (!e)
		return;

	num = e->num_hook_entries;
	ops = nf_hook_entries_get_hook_ops(e);
	head = (void *)&ops[num];
	head->allocation = e;
	call_rcu(&head->head, (*klpe___nf_hook_entries_free));
}


static struct nf_hook_entries *
(*klpe_nf_hook_entries_grow)(const struct nf_hook_entries *old,
		     const struct nf_hook_ops *reg);

static void klpr_hooks_validate(const struct nf_hook_entries *hooks)
{
#ifndef CONFIG_DEBUG_MISC
#error "klp-ccp: a preceeding branch should have been taken"
#endif
}

static struct nf_hook_entries __rcu **
(*klpe_nf_hook_entry_head)(struct net *net, int pf, unsigned int hooknum,
		   struct net_device *dev);

static int nf_ingress_check(struct net *net, const struct nf_hook_ops *reg,
			    int hooknum)
{
#ifndef CONFIG_NETFILTER_INGRESS
#error "klp-ccp: non-taken branch"
#endif
	if (reg->hooknum != hooknum ||
	    !reg->dev || dev_net(reg->dev) != net)
		return -EINVAL;

	return 0;
}

static inline bool nf_ingress_hook(const struct nf_hook_ops *reg, int pf)
{
	if ((pf == NFPROTO_NETDEV && reg->hooknum == NF_NETDEV_INGRESS) ||
	    (pf == NFPROTO_INET && reg->hooknum == NF_INET_INGRESS))
		return true;

	return false;
}

static void nf_static_key_inc(const struct nf_hook_ops *reg, int pf)
{
#ifdef CONFIG_JUMP_LABEL
	int hooknum;

	if (pf == NFPROTO_INET && reg->hooknum == NF_INET_INGRESS) {
		pf = NFPROTO_NETDEV;
		hooknum = NF_NETDEV_INGRESS;
	} else {
		hooknum = reg->hooknum;
	}
	static_key_slow_inc(&nf_hooks_needed[pf][hooknum]);
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
}

int klpp___nf_register_net_hook(struct net *net, int pf,
				  const struct nf_hook_ops *reg)
{
	struct nf_hook_entries *p, *new_hooks;
	struct nf_hook_entries __rcu **pp;
	int err;

	switch (pf) {
	case NFPROTO_NETDEV:
		err = nf_ingress_check(net, reg, NF_NETDEV_INGRESS);
		if (err < 0)
			return err;
		break;
	case NFPROTO_INET:
		if (reg->hooknum != NF_INET_INGRESS)
			break;

		err = nf_ingress_check(net, reg, NF_INET_INGRESS);
		if (err < 0)
			return err;
		break;
	}

	pp = (*klpe_nf_hook_entry_head)(net, pf, reg->hooknum, reg->dev);
	if (!pp)
		return -EINVAL;

	mutex_lock(&(*klpe_nf_hook_mutex));

	p = ({ do { } while (0 && (!((lockdep_is_held(&(*klpe_nf_hook_mutex)))))); ; ((typeof(*(*pp)) *)((*pp))); });
	new_hooks = (*klpe_nf_hook_entries_grow)(p, reg);

	if (!IS_ERR(new_hooks)) {
		klpr_hooks_validate(new_hooks);
		rcu_assign_pointer(*pp, new_hooks);
	}

	mutex_unlock(&(*klpe_nf_hook_mutex));
	if (IS_ERR(new_hooks))
		return PTR_ERR(new_hooks);

#ifdef CONFIG_NETFILTER_INGRESS
	if (nf_ingress_hook(reg, pf))
		net_inc_ingress_queue();
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	nf_static_key_inc(reg, pf);

	BUG_ON(p == new_hooks);
	klpr_nf_hook_entries_free(p);
	return 0;
}


#include "livepatch_bsc1229644.h"

#include <linux/kernel.h>
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__nf_hook_entries_free", (void *)&klpe___nf_hook_entries_free },
	{ "nf_hook_entries_grow", (void *)&klpe_nf_hook_entries_grow },
	{ "nf_hook_entry_head", (void *)&klpe_nf_hook_entry_head },
	{ "nf_hook_mutex", (void *)&klpe_nf_hook_mutex },
};

int livepatch_bsc1229644_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

