/*
 * livepatch_bsc1229553
 *
 * Fix for CVE-2024-43861, bsc#1229553
 *
 *  Upstream commit:
 *  7ab107544b77 ("net: usb: qmi_wwan: fix memory leak for not ip packets")
 *
 *  SLE12-SP5 commit:
 *  706ebe0a74ce1e4b7379e054dd0e424aea125046
 *
 *  SLE15-SP2 and -SP3 commit:
 *  5720eddd168f779d3f4fecd2dba28feb208f040a
 *
 *  SLE15-SP4 and -SP5 commit:
 *  3e796c33083281556914c9b64738f6de4a3ac33f
 *
 *  SLE15-SP6 commit:
 *  90880a5034cbb27ba579079894974a35654f1701
 *
 *  Copyright (c) 2024 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_USB_NET_QMI_WWAN)

#if !IS_MODULE(CONFIG_USB_NET_QMI_WWAN)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from drivers/net/usb/qmi_wwan.c */
#include <linux/module.h>
#include <linux/sched/signal.h>
#include <linux/netdevice.h>
#include <linux/ethtool.h>
#include <linux/etherdevice.h>

#include <linux/mii.h>

#include <linux/usb.h>

#include <linux/usb/usbnet.h>

#include <linux/u64_stats_sync.h>

struct qmi_wwan_state {
	struct usb_driver *subdriver;
	atomic_t pmcount;
	unsigned long flags;
	struct usb_interface *control;
	struct usb_interface *data;
};

enum qmi_wwan_flags {
	QMI_WWAN_FLAG_RAWIP = 1 << 0,
	QMI_WWAN_FLAG_MUX = 1 << 1,
};

struct qmimux_hdr {
	u8 pad;
	u8 mux_id;
	__be16 pkt_len;
};

struct qmimux_priv {
	struct net_device *real_dev;
	u8 mux_id;
	struct pcpu_sw_netstats __percpu *stats64;
};

static struct net_device *(*klpe_qmimux_find_dev)(struct usbnet *dev, u8 mux_id);

static int klpr_qmimux_rx_fixup(struct usbnet *dev, struct sk_buff *skb)
{
	unsigned int len, offset = 0, pad_len, pkt_len;
	struct qmimux_hdr *hdr;
	struct net_device *net;
	struct sk_buff *skbn;
	u8 qmimux_hdr_sz = sizeof(*hdr);

	while (offset + qmimux_hdr_sz < skb->len) {
		hdr = (struct qmimux_hdr *)(skb->data + offset);
		len = be16_to_cpu(hdr->pkt_len);

		/* drop the packet, bogus length */
		if (offset + len + qmimux_hdr_sz > skb->len)
			return 0;

		/* control packet, we do not know what to do */
		if (hdr->pad & 0x80)
			goto skip;

		/* extract padding length and check for valid length info */
		pad_len = hdr->pad & 0x3f;
		if (len == 0 || pad_len >= len)
			goto skip;
		pkt_len = len - pad_len;

		net = (*klpe_qmimux_find_dev)(dev, hdr->mux_id);
		if (!net)
			goto skip;
		skbn = netdev_alloc_skb(net, pkt_len);
		if (!skbn)
			return 0;
		skbn->dev = net;

		switch (skb->data[offset + qmimux_hdr_sz] & 0xf0) {
		case 0x40:
			skbn->protocol = htons(ETH_P_IP);
			break;
		case 0x60:
			skbn->protocol = htons(ETH_P_IPV6);
			break;
		default:
			/* not ip - do not know what to do */
			kfree_skb(skbn);
			goto skip;
		}

		skb_put_data(skbn, skb->data + offset + qmimux_hdr_sz, pkt_len);
		if (netif_rx(skbn) != NET_RX_SUCCESS) {
			net->stats.rx_errors++;
			return 0;
		} else {
			struct pcpu_sw_netstats *stats64;
			struct qmimux_priv *priv = netdev_priv(net);

			stats64 = this_cpu_ptr(priv->stats64);
			u64_stats_update_begin(&stats64->syncp);
			stats64->rx_packets++;
			stats64->rx_bytes += pkt_len;
			u64_stats_update_end(&stats64->syncp);
		}

skip:
		offset += len + qmimux_hdr_sz;
	}
	return 1;
}

int klpp_qmi_wwan_rx_fixup(struct usbnet *dev, struct sk_buff *skb)
{
	struct qmi_wwan_state *info = (void *)&dev->data;
	bool rawip = info->flags & QMI_WWAN_FLAG_RAWIP;
	__be16 proto;

	/* This check is no longer done by usbnet */
	if (skb->len < dev->net->hard_header_len)
		return 0;

	if (info->flags & QMI_WWAN_FLAG_MUX)
		return klpr_qmimux_rx_fixup(dev, skb);

	switch (skb->data[0] & 0xf0) {
	case 0x40:
		proto = htons(ETH_P_IP);
		break;
	case 0x60:
		proto = htons(ETH_P_IPV6);
		break;
	case 0x00:
		if (rawip)
			return 0;
		if (is_multicast_ether_addr(skb->data))
			return 1;
		/* possibly bogus destination - rewrite just in case */
		skb_reset_mac_header(skb);
		goto fix_dest;
	default:
		if (rawip)
			return 0;
		/* pass along other packets without modifications */
		return 1;
	}
	if (rawip) {
		skb_reset_mac_header(skb);
		skb->dev = dev->net; /* normally set by eth_type_trans */
		skb->protocol = proto;
		return 1;
	}

	if (skb_headroom(skb) < ETH_HLEN)
		return 0;
	skb_push(skb, ETH_HLEN);
	skb_reset_mac_header(skb);
	eth_hdr(skb)->h_proto = proto;
	eth_zero_addr(eth_hdr(skb)->h_source);
fix_dest:
	memcpy(eth_hdr(skb)->h_dest, dev->net->dev_addr, ETH_ALEN);
	return 1;
}


#include "livepatch_bsc1229553.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include "../kallsyms_relocs.h"

#define LP_MODULE "qmi_wwan"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "qmimux_find_dev", (void *)&klpe_qmimux_find_dev, "qmi_wwan" },
};

static int module_notify(struct notifier_block *nb,
			unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1229553_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LP_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1229553_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}

#endif /* IS_ENABLED(CONFIG_USB_NET_QMI_WWAN) */
