/*
 * livepatch_bsc1207190
 *
 * Fix for CVE-2023-0266, bsc#1207190
 *
 *  Upstream commit:
 *  56b88b50565c ("ALSA: pcm: Move rwsem lock inside snd_ctl_elem_read to
 *                 prevent UAF")
 *
 *  SLE12-SP4, SLE15 and SLE15-SP1 commit:
 *  not affected
 *
 *  SLE12-SP5 commit:
 *  55a788e57b910c3ff8ae11d08cbabe1dd6dbf563
 *
 *  SLE15-SP2 and -SP3 commit:
 *  90144934172236694323615b0e358e91c38b8c78
 *
 *  SLE15-SP4 commit:
 *  ffbf830983189702e7070a85fc735a6ee83a9e2a
 *
 *
 *  Copyright (c) 2023 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_SND)

#if !IS_MODULE(CONFIG_SND)
#error "Live patch supports only CONFIG_SND=m"
#endif

#if !IS_ENABLED(CONFIG_COMPAT)
#error "Live patch supports only CONFIG_COMPAT=y"
#endif

/* klp-ccp: from sound/core/control.c */
#include <linux/threads.h>
/* klp-ccp: from include/linux/interrupt.h */
#include <linux/kernel.h>
#include <linux/bitops.h>
#include <linux/bitmap.h>
#include <linux/cpumask.h>
#include <linux/hardirq.h>
#include <linux/irqflags.h>
#include <linux/hrtimer.h>
#include <linux/kref.h>
#include <linux/workqueue.h>
#include <linux/jump_label.h>
#include <linux/atomic.h>
#include <asm/ptrace.h>
#include <asm/sections.h>

/* klp-ccp: from sound/core/control.c */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/slab.h>
#include <linux/time.h>
#include <linux/mm.h>
#include <linux/math64.h>
#include <linux/sched/signal.h>
#include <sound/core.h>

/* klp-ccp: from include/sound/core.h */
#ifdef CONFIG_PM

static int (*klpe_snd_power_ref_and_wait)(struct snd_card *card);

#else /* ! CONFIG_PM */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_PM */

/* klp-ccp: from sound/core/control.c */
#include <sound/control.h>

/* klp-ccp: from include/sound/control.h */
static struct snd_kcontrol *(*klpe_snd_ctl_find_id)(struct snd_card * card, struct snd_ctl_elem_id *id);

/* klp-ccp: from sound/core/control.c */
struct snd_kctl_ioctl {
	struct list_head list;		/* list of all ioctls */
	snd_kctl_ioctl_func_t fioctl;
};

static struct rw_semaphore (*klpe_snd_ioctl_rwsem);

static struct list_head (*klpe_snd_control_ioctls);

static void (*klpe_snd_ctl_empty_read_queue)(struct snd_ctl_file * ctl);

static int (*klpe_snd_ctl_remove_user_ctl)(struct snd_ctl_file * file,
				   struct snd_ctl_elem_id *id);

static int klpr_snd_ctl_card_info(struct snd_card *card, struct snd_ctl_file * ctl,
			     unsigned int cmd, void __user *arg)
{
	struct snd_ctl_card_info *info;

	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (! info)
		return -ENOMEM;
	down_read(&(*klpe_snd_ioctl_rwsem));
	info->card = card->number;
	strscpy(info->id, card->id, sizeof(info->id));
	strscpy(info->driver, card->driver, sizeof(info->driver));
	strscpy(info->name, card->shortname, sizeof(info->name));
	strscpy(info->longname, card->longname, sizeof(info->longname));
	strscpy(info->mixername, card->mixername, sizeof(info->mixername));
	strscpy(info->components, card->components, sizeof(info->components));
	up_read(&(*klpe_snd_ioctl_rwsem));
	if (copy_to_user(arg, info, sizeof(struct snd_ctl_card_info))) {
		kfree(info);
		return -EFAULT;
	}
	kfree(info);
	return 0;
}

static int (*klpe_snd_ctl_elem_list)(struct snd_card *card,
			     struct snd_ctl_elem_list *list);

static int klpr_snd_ctl_elem_list_user(struct snd_card *card,
				  struct snd_ctl_elem_list __user *_list)
{
	struct snd_ctl_elem_list list;
	int err;

	if (copy_from_user(&list, _list, sizeof(list)))
		return -EFAULT;
	err = (*klpe_snd_ctl_elem_list)(card, &list);
	if (err)
		return err;
	if (copy_to_user(_list, &list, sizeof(list)))
		return -EFAULT;

	return 0;
}

#ifdef CONFIG_SND_CTL_VALIDATION
#error "klp-ccp: non-taken branch"
#else
static inline void fill_remaining_elem_value(struct snd_ctl_elem_value *control,
					     struct snd_ctl_elem_info *info,
					     u32 pattern)
{
}

static inline int sanity_check_elem_value(struct snd_card *card,
					  struct snd_ctl_elem_value *control,
					  struct snd_ctl_elem_info *info,
					  u32 pattern)
{
	return 0;
}
#endif

static int (*klpe_snd_ctl_elem_info_user)(struct snd_ctl_file *ctl,
				  struct snd_ctl_elem_info __user *_info);

int klpp_snd_ctl_elem_read(struct snd_card *card,
			     struct snd_ctl_elem_value *control)
{
	struct snd_kcontrol *kctl;
	struct snd_kcontrol_volatile *vd;
	unsigned int index_offset;
	struct snd_ctl_elem_info info;
	const u32 pattern = 0xdeadbeef;
	int ret;

	/*
	 * Fix CVE-2023-0266
	 *  +1 line
	 */
	down_read(&card->controls_rwsem);
	kctl = (*klpe_snd_ctl_find_id)(card, &control->id);
	/*
	 * Fix CVE-2023-0266
	 *  -2 lines, +4 lines
	 */
	if (kctl == NULL) {
		ret = -ENOENT;
		goto unlock;
	}

	index_offset = snd_ctl_get_ioff(kctl, &control->id);
	vd = &kctl->vd[index_offset];
	/*
	 * Fix CVE-2023-0266
	 *  -2 lines, +4 lines
	 */
	if (!(vd->access & SNDRV_CTL_ELEM_ACCESS_READ) || kctl->get == NULL) {
		ret = -EPERM;
		goto unlock;
	}

	snd_ctl_build_ioff(&control->id, kctl, index_offset);

#ifdef CONFIG_SND_CTL_VALIDATION
#error "klp-ccp: non-taken branch"
#endif
	if (!snd_ctl_skip_validation(&info))
		fill_remaining_elem_value(control, &info, pattern);
	ret = (*klpe_snd_power_ref_and_wait)(card);
	if (!ret)
		ret = kctl->get(kctl, control);
	snd_power_unref(card);
	if (ret < 0)
		/*
		 * Fix CVE-2023-0266
		 *  -1 line, +1 line
		 */
		goto unlock;
	if (!snd_ctl_skip_validation(&info) &&
	    sanity_check_elem_value(card, control, &info, pattern) < 0) {
		dev_err(card->dev,
			"control %i:%i:%i:%s:%i: access overflow\n",
			control->id.iface, control->id.device,
			control->id.subdevice, control->id.name,
			control->id.index);
		/*
		 * Fix CVE-2023-0266
		 *  -1 line, +2 lines
		 */
		ret = -EINVAL;
		goto unlock;
	}
/*
 * Fix CVE-2023-0266
 *  +2 lines
 */
unlock:
	up_read(&card->controls_rwsem);
	return ret;
}

static int klpp_snd_ctl_elem_read_user(struct snd_card *card,
				  struct snd_ctl_elem_value __user *_control)
{
	struct snd_ctl_elem_value *control;
	int result;

	control = memdup_user(_control, sizeof(*control));
	if (IS_ERR(control))
		return PTR_ERR(control);

	/*
	 * Fix CVE-2023-0266
	 *  -1 line
	 */
	result = klpp_snd_ctl_elem_read(card, control);
	/*
	 * Fix CVE-2023-0266
	 *  -1 line
	 */
	if (result < 0)
		goto error;

	if (copy_to_user(_control, control, sizeof(*control)))
		result = -EFAULT;
 error:
	kfree(control);
	return result;
}

static int (*klpe_snd_ctl_elem_write)(struct snd_card *card, struct snd_ctl_file *file,
			      struct snd_ctl_elem_value *control);

static int klpr_snd_ctl_elem_write_user(struct snd_ctl_file *file,
				   struct snd_ctl_elem_value __user *_control)
{
	struct snd_ctl_elem_value *control;
	struct snd_card *card;
	int result;

	control = memdup_user(_control, sizeof(*control));
	if (IS_ERR(control))
		return PTR_ERR(control);

	card = file->card;
	result = (*klpe_snd_ctl_elem_write)(card, file, control);
	if (result < 0)
		goto error;

	if (copy_to_user(_control, control, sizeof(*control)))
		result = -EFAULT;
 error:
	kfree(control);
	return result;
}

static int klpr_snd_ctl_elem_lock(struct snd_ctl_file *file,
			     struct snd_ctl_elem_id __user *_id)
{
	struct snd_card *card = file->card;
	struct snd_ctl_elem_id id;
	struct snd_kcontrol *kctl;
	struct snd_kcontrol_volatile *vd;
	int result;

	if (copy_from_user(&id, _id, sizeof(id)))
		return -EFAULT;
	down_write(&card->controls_rwsem);
	kctl = (*klpe_snd_ctl_find_id)(card, &id);
	if (kctl == NULL) {
		result = -ENOENT;
	} else {
		vd = &kctl->vd[snd_ctl_get_ioff(kctl, &id)];
		if (vd->owner != NULL)
			result = -EBUSY;
		else {
			vd->owner = file;
			result = 0;
		}
	}
	up_write(&card->controls_rwsem);
	return result;
}

static int klpr_snd_ctl_elem_unlock(struct snd_ctl_file *file,
			       struct snd_ctl_elem_id __user *_id)
{
	struct snd_card *card = file->card;
	struct snd_ctl_elem_id id;
	struct snd_kcontrol *kctl;
	struct snd_kcontrol_volatile *vd;
	int result;

	if (copy_from_user(&id, _id, sizeof(id)))
		return -EFAULT;
	down_write(&card->controls_rwsem);
	kctl = (*klpe_snd_ctl_find_id)(card, &id);
	if (kctl == NULL) {
		result = -ENOENT;
	} else {
		vd = &kctl->vd[snd_ctl_get_ioff(kctl, &id)];
		if (vd->owner == NULL)
			result = -EINVAL;
		else if (vd->owner != file)
			result = -EPERM;
		else {
			vd->owner = NULL;
			result = 0;
		}
	}
	up_write(&card->controls_rwsem);
	return result;
}

static int (*klpe_snd_ctl_elem_add_user)(struct snd_ctl_file *file,
				 struct snd_ctl_elem_info __user *_info, int replace);

static int klpr_snd_ctl_elem_remove(struct snd_ctl_file *file,
			       struct snd_ctl_elem_id __user *_id)
{
	struct snd_ctl_elem_id id;

	if (copy_from_user(&id, _id, sizeof(id)))
		return -EFAULT;
	return (*klpe_snd_ctl_remove_user_ctl)(file, &id);
}

static int klpr_snd_ctl_subscribe_events(struct snd_ctl_file *file, int __user *ptr)
{
	int subscribe;
	if (get_user(subscribe, ptr))
		return -EFAULT;
	if (subscribe < 0) {
		subscribe = file->subscribed;
		if (put_user(subscribe, ptr))
			return -EFAULT;
		return 0;
	}
	if (subscribe) {
		file->subscribed = 1;
		return 0;
	} else if (file->subscribed) {
		(*klpe_snd_ctl_empty_read_queue)(file);
		file->subscribed = 0;
	}
	return 0;
}

static int (*klpe_snd_ctl_tlv_ioctl)(struct snd_ctl_file *file,
			     struct snd_ctl_tlv __user *buf,
                             int op_flag);

long klpp_snd_ctl_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct snd_ctl_file *ctl;
	struct snd_card *card;
	struct snd_kctl_ioctl *p;
	void __user *argp = (void __user *)arg;
	int __user *ip = argp;
	int err;

	ctl = file->private_data;
	card = ctl->card;
	if (snd_BUG_ON(!card))
		return -ENXIO;
	switch (cmd) {
	case SNDRV_CTL_IOCTL_PVERSION:
		return put_user(SNDRV_CTL_VERSION, ip) ? -EFAULT : 0;
	case SNDRV_CTL_IOCTL_CARD_INFO:
		return klpr_snd_ctl_card_info(card, ctl, cmd, argp);
	case SNDRV_CTL_IOCTL_ELEM_LIST:
		return klpr_snd_ctl_elem_list_user(card, argp);
	case SNDRV_CTL_IOCTL_ELEM_INFO:
		return (*klpe_snd_ctl_elem_info_user)(ctl, argp);
	case SNDRV_CTL_IOCTL_ELEM_READ:
		return klpp_snd_ctl_elem_read_user(card, argp);
	case SNDRV_CTL_IOCTL_ELEM_WRITE:
		return klpr_snd_ctl_elem_write_user(ctl, argp);
	case SNDRV_CTL_IOCTL_ELEM_LOCK:
		return klpr_snd_ctl_elem_lock(ctl, argp);
	case SNDRV_CTL_IOCTL_ELEM_UNLOCK:
		return klpr_snd_ctl_elem_unlock(ctl, argp);
	case SNDRV_CTL_IOCTL_ELEM_ADD:
		return (*klpe_snd_ctl_elem_add_user)(ctl, argp, 0);
	case SNDRV_CTL_IOCTL_ELEM_REPLACE:
		return (*klpe_snd_ctl_elem_add_user)(ctl, argp, 1);
	case SNDRV_CTL_IOCTL_ELEM_REMOVE:
		return klpr_snd_ctl_elem_remove(ctl, argp);
	case SNDRV_CTL_IOCTL_SUBSCRIBE_EVENTS:
		return klpr_snd_ctl_subscribe_events(ctl, ip);
	case SNDRV_CTL_IOCTL_TLV_READ:
		down_read(&ctl->card->controls_rwsem);
		err = (*klpe_snd_ctl_tlv_ioctl)(ctl, argp, SNDRV_CTL_TLV_OP_READ);
		up_read(&ctl->card->controls_rwsem);
		return err;
	case SNDRV_CTL_IOCTL_TLV_WRITE:
		down_write(&ctl->card->controls_rwsem);
		err = (*klpe_snd_ctl_tlv_ioctl)(ctl, argp, SNDRV_CTL_TLV_OP_WRITE);
		up_write(&ctl->card->controls_rwsem);
		return err;
	case SNDRV_CTL_IOCTL_TLV_COMMAND:
		down_write(&ctl->card->controls_rwsem);
		err = (*klpe_snd_ctl_tlv_ioctl)(ctl, argp, SNDRV_CTL_TLV_OP_CMD);
		up_write(&ctl->card->controls_rwsem);
		return err;
	case SNDRV_CTL_IOCTL_POWER:
		return -ENOPROTOOPT;
	case SNDRV_CTL_IOCTL_POWER_STATE:
		return put_user(SNDRV_CTL_POWER_D0, ip) ? -EFAULT : 0;
	}
	down_read(&(*klpe_snd_ioctl_rwsem));
	list_for_each_entry(p, &(*klpe_snd_control_ioctls), list) {
		err = p->fioctl(card, ctl, cmd, arg);
		if (err != -ENOIOCTLCMD) {
			up_read(&(*klpe_snd_ioctl_rwsem));
			return err;
		}
	}
	up_read(&(*klpe_snd_ioctl_rwsem));
	dev_dbg(card->dev, "unknown ioctl = 0x%x\n", cmd);
	return -ENOTTY;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1207190.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "snd"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "snd_control_ioctls", (void *)&klpe_snd_control_ioctls, "snd" },
	{ "snd_ctl_elem_add_user", (void *)&klpe_snd_ctl_elem_add_user, "snd" },
	{ "snd_ctl_elem_info_user", (void *)&klpe_snd_ctl_elem_info_user,
	  "snd" },
	{ "snd_ctl_elem_list", (void *)&klpe_snd_ctl_elem_list, "snd" },
	{ "snd_ctl_elem_write", (void *)&klpe_snd_ctl_elem_write, "snd" },
	{ "snd_ctl_empty_read_queue", (void *)&klpe_snd_ctl_empty_read_queue,
	  "snd" },
	{ "snd_ctl_find_id", (void *)&klpe_snd_ctl_find_id, "snd" },
	{ "snd_ctl_remove_user_ctl", (void *)&klpe_snd_ctl_remove_user_ctl,
	  "snd" },
	{ "snd_ctl_tlv_ioctl", (void *)&klpe_snd_ctl_tlv_ioctl, "snd" },
	{ "snd_ioctl_rwsem", (void *)&klpe_snd_ioctl_rwsem, "snd" },
	{ "snd_power_ref_and_wait", (void *)&klpe_snd_power_ref_and_wait,
	  "snd" },
};

static int livepatch_bsc1207190_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1207190_module_nb = {
	.notifier_call = livepatch_bsc1207190_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1207190_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&livepatch_bsc1207190_module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LIVEPATCHED_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						  ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&livepatch_bsc1207190_module_nb);

	module_put(mod);
	return ret;
}

void livepatch_bsc1207190_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1207190_module_nb);
}

#endif /* IS_ENABLED(CONFIG_SND) */
