/*
 * bsc1203067_kvm_x86
 *
 * Fix for CVE-2022-39189, bsc#1203067 (arch/x86/kvm/x86.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_X86_64)

#if !IS_MODULE(CONFIG_KVM)
#error "Live patch supports only CONFIG_KVM=m"
#endif

#include "bsc1203067_common.h"

/* klp-ccp: from arch/x86/kvm/x86.c */
#include <linux/kvm_host.h>

/* klp-ccp: from arch/x86/include/asm/kvm_host.h */
static struct kvm_x86_ops *(*klpe_kvm_x86_ops);

static int (*klpe_kvm_set_cr8)(struct kvm_vcpu *vcpu, unsigned long cr8);

static unsigned long (*klpe_kvm_get_cr8)(struct kvm_vcpu *vcpu);

static unsigned long (*klpe_kvm_get_rflags)(struct kvm_vcpu *vcpu);

/* klp-ccp: from include/linux/kvm_host.h */
#ifdef CONFIG_KVM_ASYNC_PF

static void (*klpe_kvm_check_async_pf_completion)(struct kvm_vcpu *vcpu);

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

static void (*klpe_vcpu_load)(struct kvm_vcpu *vcpu);
static void (*klpe_vcpu_put)(struct kvm_vcpu *vcpu);

static int (*klpe_kvm_map_gfn)(struct kvm_vcpu *vcpu, gfn_t gfn, struct kvm_host_map *map,
		struct gfn_to_pfn_cache *cache, bool atomic);

static int (*klpe_kvm_unmap_gfn)(struct kvm_vcpu *vcpu, struct kvm_host_map *map,
		  struct gfn_to_pfn_cache *cache, bool dirty, bool atomic);

static void (*klpe_kvm_sigset_activate)(struct kvm_vcpu *vcpu);
static void (*klpe_kvm_sigset_deactivate)(struct kvm_vcpu *vcpu);

static void (*klpe_kvm_vcpu_block)(struct kvm_vcpu *vcpu);

int klpp_kvm_arch_vcpu_ioctl_run(struct kvm_vcpu *vcpu, struct kvm_run *kvm_run);

void klpp_kvm_arch_vcpu_put(struct kvm_vcpu *vcpu);

static int (*klpe_kvm_arch_vcpu_runnable)(struct kvm_vcpu *vcpu);

static int (*klpe_kvm_cpu_has_pending_timer)(struct kvm_vcpu *vcpu);

/* klp-ccp: from arch/x86/kvm/irq.h */
#include <linux/mm_types.h>
#include <linux/hrtimer.h>
#include <linux/kvm_host.h>
#include <linux/spinlock.h>

/* klp-ccp: from arch/x86/kvm/ioapic.h */
#include <linux/kvm_host.h>
#include <kvm/iodev.h>
/* klp-ccp: from arch/x86/kvm/lapic.h */
#include <kvm/iodev.h>
#include <linux/kvm_host.h>

static void (*klpe_kvm_apic_accept_events)(struct kvm_vcpu *vcpu);

static u64 (*klpe_kvm_get_apic_base)(struct kvm_vcpu *vcpu);

static struct static_key (*klpe_kvm_no_apic_vcpu);

static inline bool klpr_lapic_in_kernel(struct kvm_vcpu *vcpu)
{
	if (static_key_enabled(&(*klpe_kvm_no_apic_vcpu)))
		return vcpu->arch.apic;
	return true;
}

/* klp-ccp: from arch/x86/kvm/irq.h */
static inline int pic_in_kernel(struct kvm *kvm)
{
	int mode = kvm->arch.irqchip_mode;

	/* Matches smp_wmb() when setting irqchip_mode */
	smp_rmb();
	return mode == KVM_IRQCHIP_KERNEL;
}

static void (*klpe_kvm_inject_pending_timer_irqs)(struct kvm_vcpu *vcpu);

/* klp-ccp: from arch/x86/kvm/mmu.h */
#include <linux/kvm_host.h>
/* klp-ccp: from arch/x86/kvm/kvm_cache_regs.h */
#include <linux/kvm_host.h>

static inline bool is_guest_mode(struct kvm_vcpu *vcpu)
{
	return vcpu->arch.hflags & HF_GUEST_MASK;
}

static inline bool is_smm(struct kvm_vcpu *vcpu)
{
	return vcpu->arch.hflags & HF_SMM_MASK;
}

/* klp-ccp: from arch/x86/kvm/i8254.h */
#include <linux/kthread.h>
#include <kvm/iodev.h>
/* klp-ccp: from arch/x86/kvm/x86.h */
#include <linux/kvm_host.h>

/* klp-ccp: from arch/x86/kvm/cpuid.h */
#include <asm/processor.h>
/* klp-ccp: from arch/x86/kvm/pmu.h */
#include <linux/nospec.h>
/* klp-ccp: from arch/x86/kvm/hyperv.h */
#include <linux/kvm_host.h>
/* klp-ccp: from arch/x86/kvm/x86.c */
#include <linux/clocksource.h>
#include <linux/interrupt.h>
#include <linux/kvm.h>
#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/export.h>
#include <linux/moduleparam.h>
#include <linux/srcu.h>
#include <linux/slab.h>
#include <linux/perf_event.h>
#include <linux/uaccess.h>
#include <linux/hash.h>
#include <linux/pvclock_gtod.h>
#include <linux/irqbypass.h>
#include <linux/mem_encrypt.h>

/* klp-ccp: from include/trace/events/kvm.h */
KLPR_TRACE_EVENT(kvm_fpu,
	TP_PROTO(int load),
	TP_ARGS(load)
);

/* klp-ccp: from arch/x86/kvm/x86.c */
#include <asm/msr.h>
#include <asm/desc.h>
#include <linux/kernel_stat.h>
#include <asm/fpu/internal.h> /* Ugh! */
#include <asm/pvclock.h>
#include <asm/div64.h>

/* klp-ccp: from arch/x86/kvm/trace.h */
#include <asm/clocksource.h>
#include <asm/pvclock-abi.h>

/* klp-ccp: from arch/x86/kvm/trace.h */
#include <asm/vmx.h>
#include <asm/svm.h>
#include <asm/clocksource.h>
#include <asm/pvclock-abi.h>

/* klp-ccp: from arch/x86/kvm/x86.c */
static void klpr_store_regs(struct kvm_vcpu *vcpu);
static int klpr_sync_regs(struct kvm_vcpu *vcpu);

static void klpp_kvm_steal_time_set_preempted(struct kvm_vcpu *vcpu)
{
	struct kvm_host_map map;
	struct kvm_steal_time *st;

	/*
	 * Fix CVE-2022-39189
	 *  +9 lines
	 */
	/*
	 * The vCPU can be marked preempted if and only if the VM-Exit was on
	 * an instruction boundary and will not trigger guest emulation of any
	 * kind (see vcpu_run).  Vendor specific code controls (conservatively)
	 * when this is true, for example allowing the vCPU to be marked
	 * preempted if and only if the VM-Exit was due to a host interrupt.
	 */
	if (!(vcpu->arch.hflags & KLPP_HF_AT_INSN_BOUNDARY_MASK))
		return;

	if (!(vcpu->arch.st.msr_val & KVM_MSR_ENABLED))
		return;

	if (vcpu->arch.st.preempted)
		return;

	if ((*klpe_kvm_map_gfn)(vcpu, vcpu->arch.st.msr_val >> PAGE_SHIFT, &map,
			&vcpu->arch.st.cache, true))
		return;

	st = map.hva +
		offset_in_page(vcpu->arch.st.msr_val & KVM_STEAL_VALID_BITS);

	st->preempted = vcpu->arch.st.preempted = KVM_VCPU_PREEMPTED;

	(*klpe_kvm_unmap_gfn)(vcpu, &map, &vcpu->arch.st.cache, true, true);
}

void klpp_kvm_arch_vcpu_put(struct kvm_vcpu *vcpu)
{
	int idx;

	if (vcpu->preempted)
		vcpu->arch.preempted_in_kernel = !(*klpe_kvm_x86_ops)->get_cpl(vcpu);

	/*
	 * Disable page faults because we're in atomic context here.
	 * kvm_write_guest_offset_cached() would call might_fault()
	 * that relies on pagefault_disable() to tell if there's a
	 * bug. NOTE: the write to guest memory may not go through if
	 * during postcopy live migration or if there's heavy guest
	 * paging.
	 */
	pagefault_disable();
	/*
	 * kvm_memslots() will be called by
	 * kvm_write_guest_offset_cached() so take the srcu lock.
	 */
	idx = srcu_read_lock(&vcpu->kvm->srcu);
	klpp_kvm_steal_time_set_preempted(vcpu);
	srcu_read_unlock(&vcpu->kvm->srcu, idx);
	pagefault_enable();
	(*klpe_kvm_x86_ops)->vcpu_put(vcpu);
	vcpu->arch.last_host_tsc = rdtsc();
	/*
	 * If userspace has set any breakpoints or watchpoints, dr6 is restored
	 * on every vmexit, but if not, we might have a stale dr6 from the
	 * guest. do_debug expects dr6 to be cleared after it runs, do the same.
	 */
	set_debugreg(0, 6);
}

static int (*klpe_kvm_vcpu_ready_for_interrupt_injection)(struct kvm_vcpu *vcpu);

static void (*klpe_kvm_vcpu_ioctl_x86_get_vcpu_events)(struct kvm_vcpu *vcpu,
					       struct kvm_vcpu_events *events);

static int (*klpe_kvm_vcpu_ioctl_x86_set_vcpu_events)(struct kvm_vcpu *vcpu,
					      struct kvm_vcpu_events *events);

static int dm_request_for_irq_injection(struct kvm_vcpu *vcpu)
{
	return vcpu->run->request_interrupt_window &&
		likely(!pic_in_kernel(vcpu->kvm));
}

static void klpr_post_kvm_run_save(struct kvm_vcpu *vcpu)
{
	struct kvm_run *kvm_run = vcpu->run;

	kvm_run->if_flag = ((*klpe_kvm_get_rflags)(vcpu) & X86_EFLAGS_IF) != 0;
	kvm_run->flags = is_smm(vcpu) ? KVM_RUN_X86_SMM : 0;
	kvm_run->cr8 = (*klpe_kvm_get_cr8)(vcpu);
	kvm_run->apic_base = (*klpe_kvm_get_apic_base)(vcpu);
	kvm_run->ready_for_interrupt_injection =
		pic_in_kernel(vcpu->kvm) ||
		(*klpe_kvm_vcpu_ready_for_interrupt_injection)(vcpu);
}

static int (*klpe_vcpu_enter_guest)(struct kvm_vcpu *vcpu);

static inline int klpr_vcpu_block(struct kvm *kvm, struct kvm_vcpu *vcpu)
{
	if (!(*klpe_kvm_arch_vcpu_runnable)(vcpu) &&
	    (!(*klpe_kvm_x86_ops)->pre_block || (*klpe_kvm_x86_ops)->pre_block(vcpu) == 0)) {
		srcu_read_unlock(&kvm->srcu, vcpu->srcu_idx);
		(*klpe_kvm_vcpu_block)(vcpu);
		vcpu->srcu_idx = srcu_read_lock(&kvm->srcu);

		if ((*klpe_kvm_x86_ops)->post_block)
			(*klpe_kvm_x86_ops)->post_block(vcpu);

		if (!kvm_check_request(KVM_REQ_UNHALT, vcpu))
			return 1;
	}

	(*klpe_kvm_apic_accept_events)(vcpu);
	switch(vcpu->arch.mp_state) {
	case KVM_MP_STATE_HALTED:
		vcpu->arch.pv.pv_unhalted = false;
		vcpu->arch.mp_state =
			KVM_MP_STATE_RUNNABLE;
		/* fall through */
	case KVM_MP_STATE_RUNNABLE:
		vcpu->arch.apf.halted = false;
		break;
	case KVM_MP_STATE_INIT_RECEIVED:
		break;
	default:
		return -EINTR;
		break;
	}
	return 1;
}

static inline bool klpr_kvm_vcpu_running(struct kvm_vcpu *vcpu)
{
	if (is_guest_mode(vcpu) && (*klpe_kvm_x86_ops)->check_nested_events)
		(*klpe_kvm_x86_ops)->check_nested_events(vcpu, /* unused */ 0);

	return (vcpu->arch.mp_state == KVM_MP_STATE_RUNNABLE &&
		!vcpu->arch.apf.halted);
}

static int klpp_vcpu_run(struct kvm_vcpu *vcpu)
{
	int r;
	struct kvm *kvm = vcpu->kvm;

	vcpu->srcu_idx = srcu_read_lock(&kvm->srcu);
	vcpu->arch.l1tf_flush_l1d = true;

	for (;;) {
		/*
		 * Fix CVE-2022-39189
		 *  +7 lines
		 */
		/*
		 * If another guest vCPU requests a PV TLB flush in the middle
		 * of instruction emulation, the rest of the emulation could
		 * use a stale page translation. Assume that any code after
		 * this point can start executing an instruction.
		 */
		vcpu->arch.hflags &= ~KLPP_HF_AT_INSN_BOUNDARY_MASK;
		if (klpr_kvm_vcpu_running(vcpu)) {
			r = (*klpe_vcpu_enter_guest)(vcpu);
		} else {
			r = klpr_vcpu_block(kvm, vcpu);
		}

		if (r <= 0)
			break;

		kvm_clear_request(KVM_REQ_PENDING_TIMER, vcpu);
		if ((*klpe_kvm_cpu_has_pending_timer)(vcpu))
			(*klpe_kvm_inject_pending_timer_irqs)(vcpu);

		if (dm_request_for_irq_injection(vcpu) &&
			(*klpe_kvm_vcpu_ready_for_interrupt_injection)(vcpu)) {
			r = 0;
			vcpu->run->exit_reason = KVM_EXIT_IRQ_WINDOW_OPEN;
			++vcpu->stat.request_irq_exits;
			break;
		}

		(*klpe_kvm_check_async_pf_completion)(vcpu);

		if (signal_pending(current)) {
			r = -EINTR;
			vcpu->run->exit_reason = KVM_EXIT_INTR;
			++vcpu->stat.signal_exits;
			break;
		}
		if (need_resched()) {
			srcu_read_unlock(&kvm->srcu, vcpu->srcu_idx);
			cond_resched();
			vcpu->srcu_idx = srcu_read_lock(&kvm->srcu);
		}
	}

	srcu_read_unlock(&kvm->srcu, vcpu->srcu_idx);

	return r;
}

static void (*klpe_kvm_save_current_fpu)(struct fpu *fpu);

static void klpr_kvm_load_guest_fpu(struct kvm_vcpu *vcpu)
{
	fpregs_lock();

	(*klpe_kvm_save_current_fpu)(vcpu->arch.user_fpu);

	/* PKRU is separately restored in kvm_x86_ops->run.  */
	__copy_kernel_to_fpregs(&vcpu->arch.guest_fpu->state,
				~XFEATURE_MASK_PKRU);

	fpregs_mark_activate();
	fpregs_unlock();

	klpr_trace_kvm_fpu(1);
}

static void (*klpe_kvm_put_guest_fpu)(struct kvm_vcpu *vcpu);

int klpp_kvm_arch_vcpu_ioctl_run(struct kvm_vcpu *vcpu, struct kvm_run *kvm_run)
{
	int r;

	(*klpe_vcpu_load)(vcpu);
	(*klpe_kvm_sigset_activate)(vcpu);
	klpr_kvm_load_guest_fpu(vcpu);

	if (unlikely(vcpu->arch.mp_state == KVM_MP_STATE_UNINITIALIZED)) {
		if (kvm_run->immediate_exit) {
			r = -EINTR;
			goto out;
		}
		(*klpe_kvm_vcpu_block)(vcpu);
		(*klpe_kvm_apic_accept_events)(vcpu);
		kvm_clear_request(KVM_REQ_UNHALT, vcpu);
		r = -EAGAIN;
		if (signal_pending(current)) {
			r = -EINTR;
			vcpu->run->exit_reason = KVM_EXIT_INTR;
			++vcpu->stat.signal_exits;
		}
		goto out;
	}

	if (vcpu->run->kvm_valid_regs & ~KVM_SYNC_X86_VALID_FIELDS) {
		r = -EINVAL;
		goto out;
	}

	if (vcpu->run->kvm_dirty_regs) {
		r = klpr_sync_regs(vcpu);
		if (r != 0)
			goto out;
	}

	/* re-sync apic's tpr */
	if (!klpr_lapic_in_kernel(vcpu)) {
		if ((*klpe_kvm_set_cr8)(vcpu, kvm_run->cr8) != 0) {
			r = -EINVAL;
			goto out;
		}
	}

	if (unlikely(vcpu->arch.complete_userspace_io)) {
		int (*cui)(struct kvm_vcpu *) = vcpu->arch.complete_userspace_io;
		vcpu->arch.complete_userspace_io = NULL;
		r = cui(vcpu);
		if (r <= 0)
			goto out;
	} else
		WARN_ON(vcpu->arch.pio.count || vcpu->mmio_needed);

	if (kvm_run->immediate_exit)
		r = -EINTR;
	else
		r = klpp_vcpu_run(vcpu);

out:
	(*klpe_kvm_put_guest_fpu)(vcpu);
	if (vcpu->run->kvm_valid_regs)
		klpr_store_regs(vcpu);
	klpr_post_kvm_run_save(vcpu);
	(*klpe_kvm_sigset_deactivate)(vcpu);

	(*klpe_vcpu_put)(vcpu);
	return r;
}

static void (*klpe___get_regs)(struct kvm_vcpu *vcpu, struct kvm_regs *regs);

static void (*klpe___set_regs)(struct kvm_vcpu *vcpu, struct kvm_regs *regs);

static void (*klpe___get_sregs)(struct kvm_vcpu *vcpu, struct kvm_sregs *sregs);

static int (*klpe___set_sregs)(struct kvm_vcpu *vcpu, struct kvm_sregs *sregs);

static void klpr_store_regs(struct kvm_vcpu *vcpu)
{
	BUILD_BUG_ON(sizeof(struct kvm_sync_regs) > SYNC_REGS_SIZE_BYTES);

	if (vcpu->run->kvm_valid_regs & KVM_SYNC_X86_REGS)
		(*klpe___get_regs)(vcpu, &vcpu->run->s.regs.regs);

	if (vcpu->run->kvm_valid_regs & KVM_SYNC_X86_SREGS)
		(*klpe___get_sregs)(vcpu, &vcpu->run->s.regs.sregs);

	if (vcpu->run->kvm_valid_regs & KVM_SYNC_X86_EVENTS)
		(*klpe_kvm_vcpu_ioctl_x86_get_vcpu_events)(
				vcpu, &vcpu->run->s.regs.events);
}

static int klpr_sync_regs(struct kvm_vcpu *vcpu)
{
	if (vcpu->run->kvm_dirty_regs & ~KVM_SYNC_X86_VALID_FIELDS)
		return -EINVAL;

	if (vcpu->run->kvm_dirty_regs & KVM_SYNC_X86_REGS) {
		(*klpe___set_regs)(vcpu, &vcpu->run->s.regs.regs);
		vcpu->run->kvm_dirty_regs &= ~KVM_SYNC_X86_REGS;
	}
	if (vcpu->run->kvm_dirty_regs & KVM_SYNC_X86_SREGS) {
		if ((*klpe___set_sregs)(vcpu, &vcpu->run->s.regs.sregs))
			return -EINVAL;
		vcpu->run->kvm_dirty_regs &= ~KVM_SYNC_X86_SREGS;
	}
	if (vcpu->run->kvm_dirty_regs & KVM_SYNC_X86_EVENTS) {
		if ((*klpe_kvm_vcpu_ioctl_x86_set_vcpu_events)(
				vcpu, &vcpu->run->s.regs.events))
			return -EINVAL;
		vcpu->run->kvm_dirty_regs &= ~KVM_SYNC_X86_EVENTS;
	}

	return 0;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203067.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "kvm"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__get_regs", (void *)&klpe___get_regs, "kvm" },
	{ "__get_sregs", (void *)&klpe___get_sregs, "kvm" },
	{ "__set_regs", (void *)&klpe___set_regs, "kvm" },
	{ "__set_sregs", (void *)&klpe___set_sregs, "kvm" },
	{ "__tracepoint_kvm_fpu", (void *)&klpe___tracepoint_kvm_fpu, "kvm" },
	{ "kvm_apic_accept_events", (void *)&klpe_kvm_apic_accept_events,
	  "kvm" },
	{ "kvm_arch_vcpu_runnable", (void *)&klpe_kvm_arch_vcpu_runnable,
	  "kvm" },
	{ "kvm_check_async_pf_completion",
	  (void *)&klpe_kvm_check_async_pf_completion, "kvm" },
	{ "kvm_cpu_has_pending_timer", (void *)&klpe_kvm_cpu_has_pending_timer,
	  "kvm" },
	{ "kvm_get_apic_base", (void *)&klpe_kvm_get_apic_base, "kvm" },
	{ "kvm_get_cr8", (void *)&klpe_kvm_get_cr8, "kvm" },
	{ "kvm_get_rflags", (void *)&klpe_kvm_get_rflags, "kvm" },
	{ "kvm_inject_pending_timer_irqs",
	  (void *)&klpe_kvm_inject_pending_timer_irqs, "kvm" },
	{ "kvm_map_gfn", (void *)&klpe_kvm_map_gfn, "kvm" },
	{ "kvm_no_apic_vcpu", (void *)&klpe_kvm_no_apic_vcpu, "kvm" },
	{ "kvm_put_guest_fpu", (void *)&klpe_kvm_put_guest_fpu, "kvm" },
	{ "kvm_save_current_fpu", (void *)&klpe_kvm_save_current_fpu, "kvm" },
	{ "kvm_set_cr8", (void *)&klpe_kvm_set_cr8, "kvm" },
	{ "kvm_sigset_activate", (void *)&klpe_kvm_sigset_activate, "kvm" },
	{ "kvm_sigset_deactivate", (void *)&klpe_kvm_sigset_deactivate, "kvm" },
	{ "kvm_unmap_gfn", (void *)&klpe_kvm_unmap_gfn, "kvm" },
	{ "kvm_vcpu_block", (void *)&klpe_kvm_vcpu_block, "kvm" },
	{ "kvm_vcpu_ioctl_x86_get_vcpu_events",
	  (void *)&klpe_kvm_vcpu_ioctl_x86_get_vcpu_events, "kvm" },
	{ "kvm_vcpu_ioctl_x86_set_vcpu_events",
	  (void *)&klpe_kvm_vcpu_ioctl_x86_set_vcpu_events, "kvm" },
	{ "kvm_vcpu_ready_for_interrupt_injection",
	  (void *)&klpe_kvm_vcpu_ready_for_interrupt_injection, "kvm" },
	{ "kvm_x86_ops", (void *)&klpe_kvm_x86_ops, "kvm" },
	{ "vcpu_enter_guest", (void *)&klpe_vcpu_enter_guest, "kvm" },
	{ "vcpu_load", (void *)&klpe_vcpu_load, "kvm" },
	{ "vcpu_put", (void *)&klpe_vcpu_put, "kvm" },
};

static int livepatch_bsc1203067_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203067_module_nb = {
	.notifier_call = livepatch_bsc1203067_module_notify,
	.priority = INT_MIN+1,
};

int bsc1203067_kvm_x86_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1203067_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1203067_kvm_x86_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203067_module_nb);
}

#endif /* IS_ENABLED(CONFIG_X86_64) */
