/*
 * livepatch_bsc1195908
 *
 * Fix for CVE-2022-0492, bsc#1195908
 *
 *  Upstream commit:
 *  24f600856418 ("cgroup-v1: Require capabilities to set release_agent")
 *
 *  SLE12-SP3 commit:
 *  50d93c2591da9e4228629449979e6fc8dc453c5b
 *
 *  SLE12-SP4, SLE12-SP5, SLE15 and SLE15-SP1 commit:
 *  25a96a7112ab07894c6387ebb67592db9ee16315
 *
 *  SLE15-SP2 and -SP3 commit:
 *  413d689eb818bd67df51d4da49daa164e47bc89d
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from include/linux/cgroup.h */
#include <linux/cpumask.h>
#include <linux/nodemask.h>
#include <linux/rculist.h>

/* klp-ccp: from include/linux/cgroup.h */
#include <linux/fs.h>
#include <linux/seq_file.h>
#include <linux/kernfs.h>
#include <linux/jump_label.h>
#include <linux/types.h>
#include <linux/ns_common.h>

/* klp-ccp: from include/linux/cgroup.h */
#include <linux/user_namespace.h>
#include <linux/refcount.h>
#include <linux/kernel_stat.h>
#include <linux/cgroup-defs.h>
/* klp-ccp: from kernel/cgroup/cgroup-internal.h */
#include <linux/kernfs.h>
#include <linux/workqueue.h>
#include <linux/list.h>
#include <linux/refcount.h>
#include <linux/fs_context.h>

struct cgroup_fs_context {
	struct kernfs_fs_context kfc;
	struct cgroup_root	*root;
	struct cgroup_namespace	*ns;
	unsigned int	flags;			/* CGRP_ROOT_* flags */

	/* cgroup1 bits */
	bool		cpuset_clone_children;
	bool		none;			/* User explicitly requested empty subsystem */
	bool		all_ss;			/* Seen 'all' option */
	u16		subsys_mask;		/* Selected subsystems */
	char		*name;			/* Hierarchy name */
	char		*release_agent;		/* Path for release notifications */
};

static inline struct cgroup_fs_context *cgroup_fc2context(struct fs_context *fc)
{
	struct kernfs_fs_context *kfc = fc->fs_private;

	return container_of(kfc, struct cgroup_fs_context, kfc);
}

static struct cgroup_subsys *(*klpe_cgroup_subsys)[];

#define klpr_for_each_subsys(ss, ssid)                                       \
	for ((ssid) = 0; (ssid) < CGROUP_SUBSYS_COUNT &&                \
		(((ss) = (*klpe_cgroup_subsys)[ssid]) || true); (ssid)++)

static bool (*klpe_cgroup_ssid_enabled)(int ssid);

static struct cgroup *(*klpe_cgroup_kn_lock_live)(struct kernfs_node *kn, bool drain_offline);
static void (*klpe_cgroup_kn_unlock)(struct kernfs_node *kn);

static bool (*klpe_cgroup1_ssid_disabled)(int ssid);

/* klp-ccp: from kernel/cgroup/cgroup-v1.c */
#include <linux/ctype.h>
#include <linux/kmod.h>
#include <linux/mm.h>

/* klp-ccp: from kernel/cgroup/cgroup-v1.c */
#include <linux/sched/task.h>
#include <linux/magic.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/cgroupstats.h>
#include <linux/fs_parser.h>

#define cg_invalf(fc, fmt, ...) invalf(fc, fmt, ## __VA_ARGS__)

static bool (*klpe_cgroup_no_v1_named);

static spinlock_t (*klpe_release_agent_path_lock);

ssize_t klpp_cgroup_release_agent_write(struct kernfs_open_file *of,
					  char *buf, size_t nbytes, loff_t off)
{
	struct cgroup *cgrp;

	BUILD_BUG_ON(sizeof(cgrp->root->release_agent_path) < PATH_MAX);

	/*
	 * Release agent gets called with all capabilities,
	 * require capabilities to set release agent.
	 */
	if ((of->file->f_cred->user_ns != &init_user_ns) ||
		!capable(CAP_SYS_ADMIN))
		return -EPERM;

	cgrp = (*klpe_cgroup_kn_lock_live)(of->kn, false);
	if (!cgrp)
		return -ENODEV;
	spin_lock(&(*klpe_release_agent_path_lock));
	strlcpy(cgrp->root->release_agent_path, strstrip(buf),
		sizeof(cgrp->root->release_agent_path));
	spin_unlock(&(*klpe_release_agent_path_lock));
	(*klpe_cgroup_kn_unlock)(of->kn);
	return nbytes;
}

enum cgroup1_param {
	Opt_all,
	Opt_clone_children,
	Opt_cpuset_v2_mode,
	Opt_name,
	Opt_none,
	Opt_noprefix,
	Opt_release_agent,
	Opt_xattr,
};

static const struct fs_parameter_description (*klpe_cgroup1_fs_parameters);

int klpp_cgroup1_parse_param(struct fs_context *fc, struct fs_parameter *param)
{
	struct cgroup_fs_context *ctx = cgroup_fc2context(fc);
	struct cgroup_subsys *ss;
	struct fs_parse_result result;
	int opt, i;

	opt = fs_parse(fc, &(*klpe_cgroup1_fs_parameters), param, &result);
	if (opt == -ENOPARAM) {
		if (strcmp(param->key, "source") == 0) {
			if (param->type != fs_value_is_string)
				return cg_invalf(NULL, "Non-string source");
			if (fc->source)
				return cg_invalf(NULL, "Multiple sources not supported");
			fc->source = param->string;
			param->string = NULL;
			return 0;
		}
		klpr_for_each_subsys(ss, i) {
			if (strcmp(param->key, ss->legacy_name))
				continue;
			if (!(*klpe_cgroup_ssid_enabled)(i) || (*klpe_cgroup1_ssid_disabled)(i))
				return cg_invalf(NULL, "Disabled controller '%s'",
					       param->key);
			ctx->subsys_mask |= (1 << i);
			return 0;
		}
		return cg_invalf(NULL, "cgroup1: Unknown subsys name '%s'", param->key);
	}
	if (opt < 0)
		return opt;

	switch (opt) {
	case Opt_none:
		/* Explicitly have no subsystems */
		ctx->none = true;
		break;
	case Opt_all:
		ctx->all_ss = true;
		break;
	case Opt_noprefix:
		ctx->flags |= CGRP_ROOT_NOPREFIX;
		break;
	case Opt_clone_children:
		ctx->cpuset_clone_children = true;
		break;
	case Opt_cpuset_v2_mode:
		ctx->flags |= CGRP_ROOT_CPUSET_V2_MODE;
		break;
	case Opt_xattr:
		ctx->flags |= CGRP_ROOT_XATTR;
		break;
	case Opt_release_agent:
		/* Specifying two release agents is forbidden */
		if (ctx->release_agent)
			return cg_invalf(NULL, "cgroup1: release_agent respecified");
		/*
		 * Release agent gets called with all capabilities,
		 * require capabilities to set release agent.
		 */
		if ((fc->user_ns != &init_user_ns) || !capable(CAP_SYS_ADMIN))
			return cg_invalf(NULL, "cgroup1: Setting release_agent not allowed");
		ctx->release_agent = param->string;
		param->string = NULL;
		break;
	case Opt_name:
		/* blocked by boot param? */
		if ((*klpe_cgroup_no_v1_named))
			return -ENOENT;
		/* Can't specify an empty name */
		if (!param->size)
			return cg_invalf(NULL, "cgroup1: Empty name");
		if (param->size > MAX_CGROUP_ROOT_NAMELEN - 1)
			return cg_invalf(NULL, "cgroup1: Name too long");
		/* Must match [\w.-]+ */
		for (i = 0; i < param->size; i++) {
			char c = param->string[i];
			if (isalnum(c))
				continue;
			if ((c == '.') || (c == '-') || (c == '_'))
				continue;
			return cg_invalf(NULL, "cgroup1: Invalid name");
		}
		/* Specifying two names is forbidden */
		if (ctx->name)
			return cg_invalf(NULL, "cgroup1: name respecified");
		ctx->name = param->string;
		param->string = NULL;
		break;
	}
	return 0;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1195908.h"
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cgroup1_ssid_disabled", (void *)&klpe_cgroup1_ssid_disabled },
	{ "cgroup_kn_lock_live", (void *)&klpe_cgroup_kn_lock_live },
	{ "cgroup_kn_unlock", (void *)&klpe_cgroup_kn_unlock },
	{ "cgroup_ssid_enabled", (void *)&klpe_cgroup_ssid_enabled },
	{ "cgroup1_fs_parameters", (void *)&klpe_cgroup1_fs_parameters },
	{ "cgroup_no_v1_named", (void *)&klpe_cgroup_no_v1_named },
	{ "cgroup_subsys", (void *)&klpe_cgroup_subsys },
	{ "release_agent_path_lock", (void *)&klpe_release_agent_path_lock },
};

int livepatch_bsc1195908_init(void)
{
	return __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}
