/*
 * bsc1200059_kdb_main
 *
 * Fix for CVE-2022-21499, bsc#1200059 (kernel/debug/kdb/kdb_main.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_KGDB)

#include "livepatch_bsc1200059.h"
#include <linux/security.h>

/* klp-ccp: from kernel/debug/kdb/kdb_main.c */
#include <linux/types.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/kmsg_dump.h>
#include <linux/reboot.h>
#include <linux/sched.h>

/* klp-ccp: from include/linux/sched/debug.h */
static void (*klpe_show_regs)(struct pt_regs *);

/* klp-ccp: from kernel/debug/kdb/kdb_main.c */
#include <linux/smp.h>
#include <linux/vmalloc.h>
#include <linux/atomic.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/kdb.h>

/* klp-ccp: from include/linux/kdb.h */
#ifdef	CONFIG_KGDB_KDB

static const char *(*klpe_kdb_diemsg);

static int (*klpe_kdb_flags);

static int (*klpe_kdb_trap_printk);

#else /* ! CONFIG_KGDB_KDB */
#error "klp-ccp: non-taken branch"
#endif	/* CONFIG_KGDB_KDB */

/* klp-ccp: from kernel/debug/kdb/kdb_main.c */
#include <linux/notifier.h>
#include <linux/time.h>
#include <linux/sysctl.h>
#include <linux/uaccess.h>
/* klp-ccp: from kernel/debug/kdb/kdb_private.h */
#include <linux/kgdb.h>

/* klp-ccp: from kernel/debug/debug_core.h */
#ifdef CONFIG_KGDB_KDB

static int (*klpe_kdb_parse)(const char *cmdstr);

#else /* ! CONFIG_KGDB_KDB */
#error "klp-ccp: non-taken branch"
#endif /* CONFIG_KGDB_KDB */

/* klp-ccp: from kernel/debug/kdb/kdb_private.h */
#define KDB_CMD_GO	(-1001)
#define KDB_CMD_CPU	(-1002)
#define KDB_CMD_SS	(-1003)
#define KDB_CMD_KGDB (-1005)

#define KDB_DEBUG_FLAG_STATE	0x0040	/* State flags */

#define KDB_DEBUG_FLAG_SHIFT	16	/* Shift factor for dbflags */

#define KLPR_KDB_DEBUG(flag)	((*klpe_kdb_flags) &		\
	(KDB_DEBUG_FLAG_##flag << KDB_DEBUG_FLAG_SHIFT))

#define KLPR_KDB_DEBUG_STATE(text, value) if (KLPR_KDB_DEBUG(STATE)) \
		(*klpe_kdb_print_state)(text, value)

#define kdb_machreg_fmt		"0x%lx"

static char *(*klpe_kdbgetenv)(const char *);

static void (*klpe_kdb_print_state)(const char *, int);

static int (*klpe_kdb_state);

#define KDB_STATE_KDB		0x00000001	/* Cpu is inside kdb */
#define KDB_STATE_LEAVING	0x00000002	/* Cpu is leaving kdb */

#define KDB_STATE_HOLD_CPU	0x00000010	/* Hold this cpu inside kdb */
#define KDB_STATE_DOING_SS	0x00000020	/* Doing ss command */
#define KDB_STATE_SSBPT		0x00000080	/* Install breakpoint
						 * after one ss, independent of
						 * DOING_SS */
#define KDB_STATE_SUPPRESS	0x00000200	/* Suppress error messages */

#define KDB_STATE_KEYBOARD	0x00020000	/* kdb entered via
						 * keyboard on this cpu */

#define KDB_STATE_DOING_KGDB	0x00080000	/* kgdb enter now issued */

#define KLPR_KDB_STATE(flag) ((*klpe_kdb_state) & KDB_STATE_##flag)
#define KLPR_KDB_STATE_SET(flag) ((void)((*klpe_kdb_state) |= KDB_STATE_##flag))
#define KLPR_KDB_STATE_CLEAR(flag) ((void)((*klpe_kdb_state) &= ~KDB_STATE_##flag))

static int (*klpe_kdb_nextline);

#ifdef CONFIG_KGDB_KDB

int klpp_kdb_main_loop(kdb_reason_t, kdb_reason_t,
			 int, kdb_dbtrap_t, struct pt_regs *);

extern int kdb_grepping_flag;

static char (*klpe_kdb_grep_string)[];
#define KDB_GREP_STRLEN 256

static char *(*klpe_kdb_getstr)(char *, size_t, const char *);

static struct task_struct *(*klpe_kdb_curr_task)(int);

#ifdef CONFIG_KDB_KEYBOARD
static void (*klpe_kdb_kbd_cleanup_state)(void);
#else /* ! CONFIG_KDB_KEYBOARD */
#error "klp-ccp: non-taken branch"
#endif /* ! CONFIG_KDB_KEYBOARD */

static char (*klpe_kdb_prompt_str)[];

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif /* CONFIG_KGDB_KDB */

/* klp-ccp: from kernel/debug/kdb/kdb_main.c */
static int (*klpe_kdb_cmd_enabled);

extern int kdb_grepping_flag;

extern typeof(kdb_grepping_flag) kdb_grepping_flag;

static int (*klpe_kdb_go_count);

typedef struct _kdbmsg {
	int	km_diag;	/* kdb diagnostic */
	char	*km_msg;	/* Corresponding message text */
} kdbmsg_t;

static kdbmsg_t (*klpe_kdbmsgs)[22];

static const int __nkdb_err = ARRAY_SIZE((*klpe_kdbmsgs));


/* New. */
static void klpp_kdb_check_for_lockdown(void)
{
	const int write_flags = KDB_ENABLE_MEM_WRITE |
				KDB_ENABLE_REG_WRITE |
				KDB_ENABLE_FLOW_CTRL;
	const int read_flags = KDB_ENABLE_MEM_READ |
			       KDB_ENABLE_REG_READ;

	bool need_to_lockdown_write = false;
	bool need_to_lockdown_read = false;

	if ((*klpe_kdb_cmd_enabled) & (KDB_ENABLE_ALL | write_flags))
		need_to_lockdown_write =
			security_locked_down(LOCKDOWN_INTEGRITY_MAX);

	if ((*klpe_kdb_cmd_enabled) & (KDB_ENABLE_ALL | read_flags))
		need_to_lockdown_read =
			security_locked_down(LOCKDOWN_CONFIDENTIALITY_MAX);

	/* De-compose KDB_ENABLE_ALL if required */
	if (need_to_lockdown_write || need_to_lockdown_read)
		if ((*klpe_kdb_cmd_enabled) & KDB_ENABLE_ALL)
			(*klpe_kdb_cmd_enabled) = KDB_ENABLE_MASK & ~KDB_ENABLE_ALL;

	if (need_to_lockdown_write)
		(*klpe_kdb_cmd_enabled) &= ~write_flags;

	if (need_to_lockdown_read)
		(*klpe_kdb_cmd_enabled) &= ~read_flags;
}

static void klpr_kdb_cmderror(int diag)
{
	int i;

	if (diag >= 0) {
		kdb_printf("no error detected (diagnostic is %d)\n", diag);
		return;
	}

	for (i = 0; i < __nkdb_err; i++) {
		if ((*klpe_kdbmsgs)[i].km_diag == diag) {
			kdb_printf("diag: %d: %s\n", diag, (*klpe_kdbmsgs)[i].km_msg);
			return;
		}
	}

	kdb_printf("Unknown diag %d\n", -diag);
}

static bool (*klpe_defcmd_in_progress);

#define KDB_CMD_HISTORY_COUNT	32
#define CMD_BUFLEN		200	/* kdb_printf: max printline
					 * size == 256 */
static unsigned int (*klpe_cmd_head);
static unsigned int (*klpe_cmd_tail);
static unsigned int (*klpe_cmdptr);
static char (*klpe_cmd_hist)[KDB_CMD_HISTORY_COUNT][CMD_BUFLEN];
static char (*klpe_cmd_cur)[CMD_BUFLEN];

static int klpr_handle_ctrl_cmd(char *cmd)
{
#define CTRL_P	16
#define CTRL_N	14
	if ((*klpe_cmd_head) == (*klpe_cmd_tail))
		return 0;
	switch (*cmd) {
	case CTRL_P:
		if ((*klpe_cmdptr) != (*klpe_cmd_tail))
			(*klpe_cmdptr) = ((*klpe_cmdptr)-1) % KDB_CMD_HISTORY_COUNT;
		strncpy((*klpe_cmd_cur), (*klpe_cmd_hist)[(*klpe_cmdptr)], CMD_BUFLEN);
		return 1;
	case CTRL_N:
		if ((*klpe_cmdptr) != (*klpe_cmd_head))
			(*klpe_cmdptr) = ((*klpe_cmdptr)+1) % KDB_CMD_HISTORY_COUNT;
		strncpy((*klpe_cmd_cur), (*klpe_cmd_hist)[(*klpe_cmdptr)], CMD_BUFLEN);
		return 1;
	}
	return 0;
}

static void klpr_kdb_dumpregs(struct pt_regs *regs)
{
	int old_lvl = console_loglevel;
	console_loglevel = CONSOLE_LOGLEVEL_MOTORMOUTH;
	(*klpe_kdb_trap_printk)++;
	(*klpe_show_regs)(regs);
	(*klpe_kdb_trap_printk)--;
	kdb_printf("\n");
	console_loglevel = old_lvl;
}

static void drop_newline(char *buf)
{
	size_t len = strlen(buf);

	if (len == 0)
		return;
	if (*(buf + len - 1) == '\n')
		*(buf + len - 1) = '\0';
}

static int klpp_kdb_local(kdb_reason_t reason, int error, struct pt_regs *regs,
		     kdb_dbtrap_t db_result)
{
	char *cmdbuf;
	int diag;
	struct task_struct *kdb_current =
		(*klpe_kdb_curr_task)(raw_smp_processor_id());

	KLPR_KDB_DEBUG_STATE("kdb_local 1", reason);

	/*
	 * Fix CVE-2022-21499
	 *  +1 line
	 */
	klpp_kdb_check_for_lockdown();

	(*klpe_kdb_go_count) = 0;
	if (reason == KDB_REASON_DEBUG) {
		/* special case below */
	} else {
		kdb_printf("\nEntering kdb (current=0x%px, pid %d) ",
			   kdb_current, kdb_current ? kdb_current->pid : 0);
#if defined(CONFIG_SMP)
		kdb_printf("on processor %d ", raw_smp_processor_id());
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	}

	switch (reason) {
	case KDB_REASON_DEBUG:
	{
		/*
		 * If re-entering kdb after a single step
		 * command, don't print the message.
		 */
		switch (db_result) {
		case KDB_DB_BPT:
			kdb_printf("\nEntering kdb (0x%px, pid %d) ",
				   kdb_current, kdb_current->pid);
#if defined(CONFIG_SMP)
			kdb_printf("on processor %d ", raw_smp_processor_id());
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
			kdb_printf("due to Debug @ " kdb_machreg_fmt "\n",
				   instruction_pointer(regs));
			break;
		case KDB_DB_SS:
			break;
		case KDB_DB_SSBPT:
			KLPR_KDB_DEBUG_STATE("kdb_local 4", reason);
			return 1;	/* kdba_db_trap did the work */
		default:
			kdb_printf("kdb: Bad result from kdba_db_trap: %d\n",
				   db_result);
			break;
		}

	}
		break;
	case KDB_REASON_ENTER:
		if (KLPR_KDB_STATE(KEYBOARD))
			kdb_printf("due to Keyboard Entry\n");
		else
			kdb_printf("due to KDB_ENTER()\n");
		break;
	case KDB_REASON_KEYBOARD:
		KLPR_KDB_STATE_SET(KEYBOARD);
		kdb_printf("due to Keyboard Entry\n");
		break;
	case KDB_REASON_ENTER_SLAVE:
		/* drop through, slaves only get released via cpu switch */
	case KDB_REASON_SWITCH:
		kdb_printf("due to cpu switch\n");
		break;
	case KDB_REASON_OOPS:
		kdb_printf("Oops: %s\n", (*klpe_kdb_diemsg));
		kdb_printf("due to oops @ " kdb_machreg_fmt "\n",
			   instruction_pointer(regs));
		klpr_kdb_dumpregs(regs);
		break;
	case KDB_REASON_SYSTEM_NMI:
		kdb_printf("due to System NonMaskable Interrupt\n");
		break;
	case KDB_REASON_NMI:
		kdb_printf("due to NonMaskable Interrupt @ "
			   kdb_machreg_fmt "\n",
			   instruction_pointer(regs));
		break;
	case KDB_REASON_SSTEP:
	case KDB_REASON_BREAK:
		kdb_printf("due to %s @ " kdb_machreg_fmt "\n",
			   reason == KDB_REASON_BREAK ?
			   "Breakpoint" : "SS trap", instruction_pointer(regs));
		/*
		 * Determine if this breakpoint is one that we
		 * are interested in.
		 */
		if (db_result != KDB_DB_BPT) {
			kdb_printf("kdb: error return from kdba_bp_trap: %d\n",
				   db_result);
			KLPR_KDB_DEBUG_STATE("kdb_local 6", reason);
			return 0;	/* Not for us, dismiss it */
		}
		break;
	case KDB_REASON_RECURSE:
		kdb_printf("due to Recursion @ " kdb_machreg_fmt "\n",
			   instruction_pointer(regs));
		break;
	default:
		kdb_printf("kdb: unexpected reason code: %d\n", reason);
		KLPR_KDB_DEBUG_STATE("kdb_local 8", reason);
		return 0;	/* Not for us, dismiss it */
	}

	while (1) {
		/*
		 * Initialize pager context.
		 */
		(*klpe_kdb_nextline) = 1;
		KLPR_KDB_STATE_CLEAR(SUPPRESS);
		kdb_grepping_flag = 0;
		/* ensure the old search does not leak into '/' commands */
		(*klpe_kdb_grep_string)[0] = '\0';

		cmdbuf = (*klpe_cmd_cur);
		*cmdbuf = '\0';
		*((*klpe_cmd_hist)[(*klpe_cmd_head)]) = '\0';

do_full_getstr:
#if defined(CONFIG_SMP)
		snprintf((*klpe_kdb_prompt_str), CMD_BUFLEN, (*klpe_kdbgetenv)("PROMPT"),
			 raw_smp_processor_id());
#else
#error "klp-ccp: non-taken branch"
#endif
		if ((*klpe_defcmd_in_progress))
			strncat((*klpe_kdb_prompt_str), "[defcmd]", CMD_BUFLEN);

		/*
		 * Fetch command from keyboard
		 */
		cmdbuf = (*klpe_kdb_getstr)(cmdbuf, CMD_BUFLEN, (*klpe_kdb_prompt_str));
		if (*cmdbuf != '\n') {
			if (*cmdbuf < 32) {
				if ((*klpe_cmdptr) == (*klpe_cmd_head)) {
					strncpy((*klpe_cmd_hist)[(*klpe_cmd_head)], (*klpe_cmd_cur),
						CMD_BUFLEN);
					*((*klpe_cmd_hist)[(*klpe_cmd_head)] +
					  strlen((*klpe_cmd_hist)[(*klpe_cmd_head)])-1) = '\0';
				}
				if (!klpr_handle_ctrl_cmd(cmdbuf))
					*((*klpe_cmd_cur)+strlen((*klpe_cmd_cur))-1) = '\0';
				cmdbuf = (*klpe_cmd_cur);
				goto do_full_getstr;
			} else {
				strncpy((*klpe_cmd_hist)[(*klpe_cmd_head)], (*klpe_cmd_cur),
					CMD_BUFLEN);
			}

			(*klpe_cmd_head) = ((*klpe_cmd_head)+1) % KDB_CMD_HISTORY_COUNT;
			if ((*klpe_cmd_head) == (*klpe_cmd_tail))
				(*klpe_cmd_tail) = ((*klpe_cmd_tail)+1) % KDB_CMD_HISTORY_COUNT;
		}

		(*klpe_cmdptr) = (*klpe_cmd_head);
		diag = (*klpe_kdb_parse)(cmdbuf);
		if (diag == KDB_NOTFOUND) {
			drop_newline(cmdbuf);
			kdb_printf("Unknown kdb command: '%s'\n", cmdbuf);
			diag = 0;
		}
		if (diag == KDB_CMD_GO
		 || diag == KDB_CMD_CPU
		 || diag == KDB_CMD_SS
		 || diag == KDB_CMD_KGDB)
			break;

		if (diag)
			klpr_kdb_cmderror(diag);
	}
	KLPR_KDB_DEBUG_STATE("kdb_local 9", diag);
	return diag;
}

int klpp_kdb_main_loop(kdb_reason_t reason, kdb_reason_t reason2, int error,
	      kdb_dbtrap_t db_result, struct pt_regs *regs)
{
	int result = 1;
	/* Stay in kdb() until 'go', 'ss[b]' or an error */
	while (1) {
		/*
		 * All processors except the one that is in control
		 * will spin here.
		 */
		KLPR_KDB_DEBUG_STATE("kdb_main_loop 1", reason);
		while (KLPR_KDB_STATE(HOLD_CPU)) {
			/* state KDB is turned off by kdb_cpu to see if the
			 * other cpus are still live, each cpu in this loop
			 * turns it back on.
			 */
			if (!KLPR_KDB_STATE(KDB))
				KLPR_KDB_STATE_SET(KDB);
		}

		KLPR_KDB_STATE_CLEAR(SUPPRESS);
		KLPR_KDB_DEBUG_STATE("kdb_main_loop 2", reason);
		if (KLPR_KDB_STATE(LEAVING))
			break;	/* Another cpu said 'go' */
		/* Still using kdb, this processor is in control */
		result = klpp_kdb_local(reason2, error, regs, db_result);
		KLPR_KDB_DEBUG_STATE("kdb_main_loop 3", result);

		if (result == KDB_CMD_CPU)
			break;

		if (result == KDB_CMD_SS) {
			KLPR_KDB_STATE_SET(DOING_SS);
			break;
		}

		if (result == KDB_CMD_KGDB) {
			if (!KLPR_KDB_STATE(DOING_KGDB))
				kdb_printf("Entering please attach debugger "
					   "or use $D#44+ or $3#33\n");
			break;
		}
		if (result && result != 1 && result != KDB_CMD_GO)
			kdb_printf("\nUnexpected kdb_local return code %d\n",
				   result);
		KLPR_KDB_DEBUG_STATE("kdb_main_loop 4", reason);
		break;
	}
	if (KLPR_KDB_STATE(DOING_SS))
		KLPR_KDB_STATE_CLEAR(SSBPT);

	/* Clean up any keyboard devices before leaving */
	(*klpe_kdb_kbd_cleanup_state)();

	return result;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "bsc1200059_common.h"
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cmd_cur", (void *)&klpe_cmd_cur },
	{ "cmd_head", (void *)&klpe_cmd_head },
	{ "cmd_hist", (void *)&klpe_cmd_hist },
	{ "cmd_tail", (void *)&klpe_cmd_tail },
	{ "cmdptr", (void *)&klpe_cmdptr },
	{ "defcmd_in_progress", (void *)&klpe_defcmd_in_progress },
	{ "kdb_cmd_enabled", (void *)&klpe_kdb_cmd_enabled },
	{ "kdb_curr_task", (void *)&klpe_kdb_curr_task },
	{ "kdb_diemsg", (void *)&klpe_kdb_diemsg },
	{ "kdb_flags", (void *)&klpe_kdb_flags },
	{ "kdb_getstr", (void *)&klpe_kdb_getstr },
	{ "kdb_go_count", (void *)&klpe_kdb_go_count },
	{ "kdb_grep_string", (void *)&klpe_kdb_grep_string },
	{ "kdb_kbd_cleanup_state", (void *)&klpe_kdb_kbd_cleanup_state },
	{ "kdb_nextline", (void *)&klpe_kdb_nextline },
	{ "kdb_parse", (void *)&klpe_kdb_parse },
	{ "kdb_print_state", (void *)&klpe_kdb_print_state },
	{ "kdb_prompt_str", (void *)&klpe_kdb_prompt_str },
	{ "kdb_state", (void *)&klpe_kdb_state },
	{ "kdb_trap_printk", (void *)&klpe_kdb_trap_printk },
	{ "kdbgetenv", (void *)&klpe_kdbgetenv },
	{ "kdbmsgs", (void *)&klpe_kdbmsgs },
	{ "show_regs", (void *)&klpe_show_regs },
};

int livepatch_bsc1200059_kdb_main_init(void)
{
	return __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

#endif /* IS_ENABLED(CONFIG_KGDB) */
