# -*- coding: utf-8 -*-
#
# Copyright (c) 2014 SUSE LLC
#
# This software is licensed to you under the GNU General Public License,
# version 2 (GPLv2). There is NO WARRANTY for this software, express or
# implied, including the implied warranties of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. You should have received a copy of GPLv2
# along with this software; if not, see
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt.
#
# SUSE trademarks are not licensed under GPLv2. No permission is
# granted to use or replicate SUSE trademarks that are incorporated
# in this software or its documentation.

from enum import Enum

from spacewalk.susemanager.mgr_sync.channel import Channel


class Product(object):

    class Status(str, Enum):
        INSTALLED = "INSTALLED"
        AVAILABLE = "AVAILABLE"
        UNAVAILABLE = "UNAVAILABLE"

    def __init__(self, data):
        self.arch = data["arch"]
        self.friendly_name = data["friendly_name"]
        self.status = Product.Status(data["status"].upper())
        self.extensions = []
        self._parse_extensions(data["extensions"])
        self.channels = [Channel(channel) for channel in data['channels']]

    @property
    def short_status(self):
        # pylint: disable=E1101
        if self.status == Product.Status.AVAILABLE:
            return "[ ]"
        else:
            return "[%s]" % str(self.status.value)[0]

    def to_ascii_row(self):
        return "{0} {1} ({2})".format(self.short_status, self.friendly_name,
                                      self.arch)

    def to_stdout(self, indentation_level=0, filter=None, expand=False,
                  interactive_data={}):
        prefix = indentation_level * "  "
        if interactive_data:
            if self.status in (Product.Status.INSTALLED,
                               Product.Status.UNAVAILABLE):
                prefix = "     " + prefix
            else:
                counter = interactive_data['counter']
                prefix = "{0:03}) {1}".format(counter, prefix)
                interactive_data['num_prod'][counter] = self
                interactive_data['counter'] += 1

        if not filter or self.matches_filter(filter):
            print(prefix + self.to_ascii_row())

            if (not expand and self.status is not Product.Status.INSTALLED) or \
               self.status == Product.Status.UNAVAILABLE:
                return

            indentation_level += 1
            for ext in self.extensions:
                ext.to_stdout(indentation_level=indentation_level,
                              expand=expand,
                              filter=filter,
                              interactive_data=interactive_data)

    def _parse_extensions(self, data):
        for extension in data:
            self.extensions.append(Product(extension))

    def matches_filter(self, filter):
        if not self.extensions:
            return filter in self.to_ascii_row().lower()
        else:
            for ext in self.extensions:
                if ext.matches_filter(filter):
                    return True

        return False


def parse_products(data, log):
    """
    Parses the data returned by SUSE Manager list products API.
    Returns a list of the Products.
    """
    log.info("Parsing products...")

    products = []

    for p in data:
        log.debug("Found product '{0} {1}'".format(
            Product(p).friendly_name, Product(p).arch))
        products.append(Product(p))

    return products
