/**
 * Copyright (c) 2009--2012 Red Hat, Inc.
 *
 * This software is licensed to you under the GNU General Public License,
 * version 2 (GPLv2). There is NO WARRANTY for this software, express or
 * implied, including the implied warranties of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. You should have received a copy of GPLv2
 * along with this software; if not, see
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt.
 *
 * Red Hat trademarks are not licensed under GPLv2. No permission is
 * granted to use or replicate Red Hat trademarks that are incorporated
 * in this software or its documentation.
 */
package com.redhat.rhn.taskomatic.task;

import com.redhat.rhn.common.conf.Config;
import com.redhat.rhn.common.conf.ConfigDefaults;
import com.redhat.rhn.domain.channel.Channel;
import com.redhat.rhn.domain.channel.ChannelFactory;

import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Used for syncing repos (like yum repos) to a channel.
 * This really just calls a python script.
 */
public class RepoSyncTask extends RhnJavaJob {

    /**
     * {@inheritDoc}
     */
    @Override
    @SuppressWarnings("unchecked")
    public void execute(JobExecutionContext context) {
        List<Long> channelIds = getChannelIds(context.getJobDetail().getJobDataMap());

        String[] lparams = {"no-errata", "sync-kickstart", "fail"};
        List<String> ltrue = Arrays.asList("true", "1");
        String params = "";

        for (String p : lparams) {
            if (context.getJobDetail().getJobDataMap().containsKey(p)) {
                if (ltrue.contains(context.getJobDetail().getJobDataMap().get(p).toString()
                        .toLowerCase().trim())) {
                    params = params + " --" + p;
                }
            }
        }

        for (Long channelId : channelIds) {
            Channel channel = ChannelFactory.lookupById(channelId);
            if (channel != null) {
                log.info("Syncing repos for channel: " + channel.getName());

                try {
                    executeExtCmd(getSyncCommand(channel, params).toArray(new String[0]));
                    channel.setLastSynced(new Date());
                }
                catch (JobExecutionException e) {
                    log.error(e.getMessage());
                }
            }
            else {
                log.error("No such channel with channel_id " + channelId);
            }
        }
    }

    /**
     * Gets the ids of channel(s) in a schedule/job data map.
     *
     * @param dataMap the data map
     * @return the channel ids
     */
    @SuppressWarnings("unchecked")
    public static List<Long> getChannelIds(final Map<String, Object> dataMap) {
        List<Long> result = new LinkedList<>();
        if (dataMap != null) {
            // bulk reposync case
            List<String> channelIds = (List<String>) dataMap.get("channel_ids");

            // single reposync case
            if (channelIds == null) {
                channelIds = new LinkedList<>();
                String channelId = (String) dataMap.get("channel_id");
                if (channelId != null) {
                    channelIds.add(channelId);
                }
            }

            // String -> Long
            for (String channelId : channelIds) {
                try {
                  result.add(Long.parseLong(channelId));
                }
                catch (NumberFormatException nfe) {
                    // there is a channel id but it is not valid
                }
            }
        }
        return result;
    }

    private static List<String> getSyncCommand(Channel c, String params) {
        List<String> cmd = new ArrayList<String>();
        cmd.add(Config.get().getString(ConfigDefaults.SPACEWALK_REPOSYNC_PATH,
                "/usr/bin/spacewalk-repo-sync"));
        cmd.add("--channel");
        cmd.add(c.getLabel());
        cmd.add("--type");
        cmd.add(ChannelFactory.CONTENT_SOURCE_TYPE_YUM.getLabel());
        cmd.add("--non-interactive");
        cmd.add(params);
        return cmd;
    }
}
