/**
 * Copyright (c) 2014 SUSE LLC
 *
 * This software is licensed to you under the GNU General Public License,
 * version 2 (GPLv2). There is NO WARRANTY for this software, express or
 * implied, including the implied warranties of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. You should have received a copy of GPLv2
 * along with this software; if not, see
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt.
 *
 * Red Hat trademarks are not licensed under GPLv2. No permission is
 * granted to use or replicate Red Hat trademarks that are incorporated
 * in this software or its documentation.
 */
package com.redhat.rhn.manager.setup;

import com.redhat.rhn.common.db.datasource.ModeFactory;
import com.redhat.rhn.common.db.datasource.SelectMode;
import com.redhat.rhn.domain.channel.ChannelFactory;
import com.redhat.rhn.manager.channel.ChannelManager;
import com.redhat.rhn.manager.content.MgrSyncUtils;
import com.redhat.rhn.manager.satellite.Executor;
import com.redhat.rhn.taskomatic.TaskoFactory;
import com.redhat.rhn.taskomatic.TaskoRun;
import com.redhat.rhn.taskomatic.TaskoSchedule;
import com.redhat.rhn.taskomatic.task.RepoSyncTask;
import com.redhat.rhn.taskomatic.task.TaskConstants;

import com.suse.manager.model.products.Channel;
import com.suse.manager.model.products.Product;
import com.suse.manager.model.products.Product.SyncStatus;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Manager class for interacting with SUSE products.
 */
public abstract class ProductSyncManager {

    /** The logger. */
    protected static Logger logger = Logger.getLogger(ProductSyncManager.class);

    /**
     * Create {@link ProductSyncManager} instance for a given {@link Executor}.
     * @param executorIn instance of {@link Executor}
     * @return instance of {@link ProductSyncManager}
     */
    public static ProductSyncManager createInstance(Executor executorIn) {
        return MgrSyncUtils.isMigratedToSCC() ? new SCCProductSyncManager() :
                new NCCProductSyncManager(executorIn);
    }

    /**
     * Create {@link ProductSyncManager} instance.
     * @return instance of {@link ProductSyncManager}
     */
    public static ProductSyncManager createInstance() {
        return MgrSyncUtils.isMigratedToSCC() ? new SCCProductSyncManager() :
            new NCCProductSyncManager();
    }

    /**
     * Returns a list of base products.
     * @return the products list
     * @throws ProductSyncException if an error occurred
     */
    public abstract List<Product> getBaseProducts() throws ProductSyncException;

    /**
     * Adds multiple products.
     * @param productIdents the product ident list
     * @throws ProductSyncException if an error occurred
     */
    public void addProducts(List<String> productIdents) throws ProductSyncException {
        for (String productIdent : productIdents) {
            addProduct(productIdent);
        }
    }

    /**
     * Adds the product.
     * @param productIdent the product ident
     * @throws ProductSyncException if an error occurred
     */
    public abstract void addProduct(String productIdent) throws ProductSyncException;

    /**
     * Refresh product, channel and subscription information without triggering
     * any reposysnc.
     * @throws ProductSyncException if the refresh failed
     * @throws InvalidMirrorCredentialException if mirror credentials are not valid
     * @throws ConnectionException if a connection to NCC was not possible
     */
    public abstract void refreshProducts() throws ProductSyncException,
            InvalidMirrorCredentialException, ConnectionException;

    /**
     * Get the synchronization status for a given product.
     * @param product product
     * @return sync status as string
     */
    protected SyncStatus getProductSyncStatus(Product product) {
        // Compute statistics about channels
        int finishedCounter = 0;
        int failedCounter = 0;
        Date maxLastSyncDate = null;
        StringBuilder debugDetails = new StringBuilder();

        for (Channel c : product.getMandatoryChannels()) {
            SyncStatus channelStatus = getChannelSyncStatus(c);

            if (StringUtils.isNotBlank(channelStatus.getDetails())) {
                debugDetails.append(channelStatus.getDetails());
            }

            if (channelStatus.isFinished()) {
                logger.debug("Channel finished: " + c.getLabel());
                finishedCounter++;
            }
            else if (channelStatus.isFailed()) {
                logger.debug("Channel failed: " + c.getLabel());
                failedCounter++;
            }
            else {
                logger.debug("Channel in progress: " + c.getLabel());
            }

            Date lastSyncDate = channelStatus.getLastSyncDate();
            if (maxLastSyncDate == null ||
                    (lastSyncDate != null && lastSyncDate.after(maxLastSyncDate))) {
                maxLastSyncDate = lastSyncDate;
            }
        }

        // Set FINISHED if all mandatory channels have metadata
        if (finishedCounter == product.getMandatoryChannels().size()) {
            SyncStatus result = new SyncStatus(SyncStatus.SyncStage.FINISHED);
            result.setLastSyncDate(maxLastSyncDate);
            return result;
        }
        // Status is FAILED if at least one channel has failed
        else if (failedCounter > 0) {
            SyncStatus failedResult = new SyncStatus(SyncStatus.SyncStage.FAILED);
            failedResult.setDetails(debugDetails.toString());
            return failedResult;
        }
        // Otherwise return IN_PROGRESS
        else {
            SyncStatus status = new SyncStatus(SyncStatus.SyncStage.IN_PROGRESS);
            int totalChannels = product.getMandatoryChannels().size();
            status.setSyncProgress((finishedCounter * 100) / totalChannels);
            return status;
        }
    }

    /**
     * Get the synchronization status for a given channel.
     * @param channel the channel
     * @return channel sync status as string
     */
    private SyncStatus getChannelSyncStatus(Channel channel) {
        // Fall back to FAILED if no progress or success is detected
        SyncStatus channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.FAILED);

        // Check for success: is there metadata for this channel?
        com.redhat.rhn.domain.channel.Channel c =
                ChannelFactory.lookupByLabel(channel.getLabel());

        // the XML data may say P, but if the channel is not in the database
        // we assume the XML data is wrong
        if (c == null) {
            return new SyncStatus(SyncStatus.SyncStage.NOT_MIRRORED);
        }

        if (ChannelManager.getRepoLastBuild(c) != null) {
            channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.FINISHED);
            channelSyncStatus.setLastSyncDate(c.getLastSynced());
            return channelSyncStatus;
        }

        // No success (metadata), check for jobs in taskomatic
        List<TaskoRun> runs = TaskoFactory.listRunsByBunch("repo-sync-bunch");
        boolean repoSyncRunFound = false;
        Date lastRunEndTime = null;
        Long channelId = c.getId();

        // Runs are sorted by start time, recent ones first
        for (TaskoRun run : runs) {
            // Get the channel id of that run
            TaskoSchedule schedule = TaskoFactory.lookupScheduleById(run.getScheduleId());
            List<Long> scheduleChannelIds =
                RepoSyncTask.getChannelIds(schedule.getDataMap());

            if (scheduleChannelIds.contains(channelId)) {
                // We found a repo-sync run for this channel
                repoSyncRunFound = true;
                lastRunEndTime = run.getEndTime();

                // Get debug information
                String debugInfo = run.getTailOfStdError(1024);
                if (debugInfo.isEmpty()) {
                    debugInfo = run.getTailOfStdOutput(1024);
                }

                // Set the status and debug info
                String runStatus = run.getStatus();
                if (logger.isDebugEnabled()) {
                    logger.debug("Repo sync run found for channel " + c.getLabel() +
                            " (" + runStatus + ")");
                }

                String prefix = "setupwizard.syncstatus.";
                if (runStatus.equals(TaskoRun.STATUS_FAILED) ||
                        runStatus.equals(TaskoRun.STATUS_INTERRUPTED)) {
                    // Reposync has failed or has been interrupted
                    channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.FAILED);
                    channelSyncStatus.setMessageKey(prefix + "message.reposync.failed");
                    channelSyncStatus.setDetails(debugInfo);
                    // Don't return from here, there might be a new schedule already
                }
                else if (runStatus.equals(TaskoRun.STATUS_READY_TO_RUN) ||
                        runStatus.equals(TaskoRun.STATUS_RUNNING)) {
                    // Reposync is in progress
                    channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.IN_PROGRESS);
                    channelSyncStatus.setMessageKey(prefix + "message.reposync.progress");
                    channelSyncStatus.setDetails(debugInfo);
                    return channelSyncStatus;
                }

                // We look at the latest run only
                break;
            }
        }

        // Check if there is a schedule that is newer than the last (FAILED) run
        if (!repoSyncRunFound || channelSyncStatus.isFailed()) {
            List<TaskoSchedule> schedules =
                    TaskoFactory.listRepoSyncSchedulesNewerThan(lastRunEndTime);
            for (TaskoSchedule s : schedules) {
                List<Long> scheduleChannelIds =
                        RepoSyncTask.getChannelIds(s.getDataMap());
                if (scheduleChannelIds.contains(channelId)) {
                    // There is a schedule for this channel
                    channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.IN_PROGRESS);
                    return channelSyncStatus;
                }
            }

            // No schedule found, return FAILED
            return channelSyncStatus;
        }

        // Check if channel metadata generation is in progress
        if (ChannelManager.isChannelLabelInProgress(channel.getLabel())) {
            channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.IN_PROGRESS);
            return channelSyncStatus;
        }

        // Check for queued items (merge this with the above method?)
        SelectMode selector = ModeFactory.getMode(TaskConstants.MODE_NAME,
                TaskConstants.TASK_QUERY_REPOMD_CANDIDATES_DETAILS_QUERY);
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("channel_label", channel.getLabel());
        if (selector.execute(params).size() > 0) {
            channelSyncStatus = new SyncStatus(SyncStatus.SyncStage.IN_PROGRESS);
            return channelSyncStatus;
        }

        // Otherwise return FAILED
        return channelSyncStatus;
    }
}
