/*
 * bsc1195308_link
 *
 * Fix for CVE-2022-0435, bsc#1195308 (net/tipc/link.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if !IS_MODULE(CONFIG_TIPC)
#error "Live patch supports only CONFIG_TIPC=m"
#endif

#include "bsc1195308_common.h"

#include <linux/kernel.h>
#include <linux/tracepoint.h>

/* from include/linux/tracepoint.h */
#define KLPR___DECLARE_TRACE(name, proto, args, cond, data_proto, data_args) \
	static struct tracepoint (*klpe___tracepoint_##name);		\
	static inline void klpr_trace_##name(proto)			\
	{								\
		if (unlikely(static_key_enabled(&(*klpe___tracepoint_##name).key))) \
			__DO_TRACE(&(*klpe___tracepoint_##name),	\
				TP_PROTO(data_proto),			\
				TP_ARGS(data_args),			\
				TP_CONDITION(cond), 0);		\
		if (IS_ENABLED(CONFIG_LOCKDEP) && (cond)) {		\
			rcu_read_lock_sched_notrace();			\
			rcu_dereference_sched((*klpe___tracepoint_##name).funcs); \
			rcu_read_unlock_sched_notrace();		\
		}							\
	}								\

#define KLPR_DECLARE_TRACE(name, proto, args)				\
	KLPR___DECLARE_TRACE(name, PARAMS(proto), PARAMS(args),		\
			cpu_online(raw_smp_processor_id()),		\
			PARAMS(void *__data, proto),			\
			PARAMS(__data, args))

#define KLPR_DEFINE_EVENT(template, name, proto, args)		\
	KLPR_DECLARE_TRACE(name, PARAMS(proto), PARAMS(args))

/* klp-ccp: from net/tipc/core.h */
static unsigned int (*klpe_tipc_net_id) __read_mostly;

static inline struct tipc_net *klpr_tipc_net(struct net *net)
{
	return net_generic(net, (*klpe_tipc_net_id));
}

/* klp-ccp: from net/tipc/msg.h */
static bool (*klpe_tipc_msg_extract)(struct sk_buff *skb, struct sk_buff **iskb, int *pos);

static void (*klpe___tipc_skb_queue_sorted)(struct sk_buff_head *list, u16 seqno,
			     struct sk_buff *skb);

/* klp-ccp: from net/tipc/addr.h */
static inline u32 klpr_tipc_own_addr(struct net *net)
{
	return klpr_tipc_net(net)->node_addr;
}

/* klp-ccp: from net/tipc/link.h */
static int (*klpe_tipc_link_fsm_evt)(struct tipc_link *l, int evt);

static bool (*klpe_tipc_link_is_synching)(struct tipc_link *l);

static bool (*klpe_tipc_link_is_blocked)(struct tipc_link *l);

static bool (*klpe_tipc_link_validate_msg)(struct tipc_link *l, struct tipc_msg *hdr);

int klpp_tipc_link_rcv(struct tipc_link *l, struct sk_buff *skb,
		  struct sk_buff_head *xmitq);
static int (*klpe_tipc_link_build_state_msg)(struct tipc_link *l, struct sk_buff_head *xmitq);

/* klp-ccp: from net/tipc/trace.h */
#include <linux/tracepoint.h>

enum {
	TIPC_DUMP_NONE		= 0,

	TIPC_DUMP_TRANSMQ	= 1,
	TIPC_DUMP_BACKLOGQ	= (1 << 1),
	TIPC_DUMP_DEFERDQ	= (1 << 2),
	TIPC_DUMP_INPUTQ	= (1 << 3),
	TIPC_DUMP_WAKEUP        = (1 << 4),

	TIPC_DUMP_SK_SNDQ	= (1 << 8),
	TIPC_DUMP_SK_RCVQ	= (1 << 9),
	TIPC_DUMP_SK_BKLGQ	= (1 << 10),
	TIPC_DUMP_ALL		= 0xffffu
};

#define KLPR_DEFINE_SKB_EVENT(name) \
KLPR_DEFINE_EVENT(tipc_skb_class, name, \
	TP_PROTO(struct sk_buff *skb, bool more, const char *header), \
	TP_ARGS(skb, more, header))
KLPR_DEFINE_SKB_EVENT(tipc_skb_dump);

KLPR_DEFINE_SKB_EVENT(tipc_proto_rcv);

#define KLPR_DEFINE_LINK_EVENT(name) \
KLPR_DEFINE_EVENT(tipc_link_class, name, \
	TP_PROTO(struct tipc_link *l, u16 dqueues, const char *header), \
	TP_ARGS(l, dqueues, header))
KLPR_DEFINE_LINK_EVENT(tipc_link_dump);

/* klp-ccp: from net/tipc/link.c */
#include <linux/pkt_sched.h>

struct tipc_stats {
	u32 sent_pkts;
	u32 recv_pkts;
	u32 sent_states;
	u32 recv_states;
	u32 sent_probes;
	u32 recv_probes;
	u32 sent_nacks;
	u32 recv_nacks;
	u32 sent_acks;
	u32 sent_bundled;
	u32 sent_bundles;
	u32 recv_bundled;
	u32 recv_bundles;
	u32 retransmitted;
	u32 sent_fragmented;
	u32 sent_fragments;
	u32 recv_fragmented;
	u32 recv_fragments;
	u32 link_congs;		/* # port sends blocked by congestion */
	u32 deferred_recv;
	u32 duplicates;
	u32 max_queue_sz;	/* send queue size high water mark */
	u32 accu_queue_sz;	/* used for send queue size profiling */
	u32 queue_sz_counts;	/* used for send queue size profiling */
	u32 msg_length_counts;	/* used for message length profiling */
	u32 msg_lengths_total;	/* used for message length profiling */
	u32 msg_length_profile[7]; /* used for msg. length profiling */
};

struct tipc_link {
	u32 addr;
	char name[TIPC_MAX_LINK_NAME];
	struct net *net;

	/* Management and link supervision data */
	u16 peer_session;
	u16 session;
	u16 snd_nxt_state;
	u16 rcv_nxt_state;
	u32 peer_bearer_id;
	u32 bearer_id;
	u32 tolerance;
	u32 abort_limit;
	u32 state;
	u16 peer_caps;
	bool in_session;
	bool active;
	u32 silent_intv_cnt;
	char if_name[TIPC_MAX_IF_NAME];
	u32 priority;
	char net_plane;
	struct tipc_mon_state mon_state;
	u16 rst_cnt;

	/* Failover/synch */
	u16 drop_point;
	struct sk_buff *failover_reasm_skb;
	struct sk_buff_head failover_deferdq;

	/* Max packet negotiation */
	u16 mtu;
	u16 advertised_mtu;

	/* Sending */
	struct sk_buff_head transmq;
	struct sk_buff_head backlogq;
	struct {
		u16 len;
		u16 limit;
		struct sk_buff *target_bskb;
	} backlog[5];
	u16 snd_nxt;
	u16 window;

	/* Reception */
	u16 rcv_nxt;
	u32 rcv_unacked;
	struct sk_buff_head deferdq;
	struct sk_buff_head *inputq;
	struct sk_buff_head *namedq;

	/* Congestion handling */
	struct sk_buff_head wakeupq;

	/* Fragmentation/reassembly */
	struct sk_buff *reasm_buf;

	/* Broadcast */
	u16 ackers;
	u16 acked;
	struct tipc_link *bc_rcvlink;
	struct tipc_link *bc_sndlink;
	u8 nack_state;
	bool bc_peer_is_up;

	/* Statistics */
	struct tipc_stats stats;
};

#define TIPC_UC_RETR_TIME (jiffies + msecs_to_jiffies(1))

enum {
	LINK_ESTABLISHED     = 0xe,
	LINK_ESTABLISHING    = 0xe  << 4,
	LINK_RESET           = 0x1  << 8,
	LINK_RESETTING       = 0x2  << 12,
	LINK_PEER_RESET      = 0xd  << 16,
	LINK_FAILINGOVER     = 0xf  << 20,
	LINK_SYNCHING        = 0xc  << 24
};

static int link_is_up(struct tipc_link *l)
{
	return l->state & (LINK_ESTABLISHED | LINK_SYNCHING);
}

static int klpp_tipc_link_proto_rcv(struct tipc_link *l, struct sk_buff *skb,
			       struct sk_buff_head *xmitq);
static void (*klpe_tipc_link_build_proto_msg)(struct tipc_link *l, int mtyp, bool probe,
				      bool probe_reply, u16 rcvgap,
				      int tolerance, int priority,
				      struct sk_buff_head *xmitq);

static int klpr_tipc_link_build_nack_msg(struct tipc_link *l,
				    struct sk_buff_head *xmitq);

static bool tipc_link_release_pkts(struct tipc_link *l, u16 to);

static int klpr_tipc_link_advance_transmq(struct tipc_link *l, u16 acked, u16 gap,
				     struct tipc_gap_ack_blks *ga,
				     struct sk_buff_head *xmitq);

static bool link_is_bc_sndlink(struct tipc_link *l)
{
	return !l->bc_sndlink;
}

static bool link_is_bc_rcvlink(struct tipc_link *l)
{
	return ((l->bc_rcvlink == l) && !link_is_bc_sndlink(l));
}

static void (*klpe_link_prepare_wakeup)(struct tipc_link *l);

static void (*klpe_tipc_link_advance_backlog)(struct tipc_link *l,
				      struct sk_buff_head *xmitq);

static bool (*klpe_link_retransmit_failure)(struct tipc_link *l, struct tipc_link *r,
				    int *rc);

static bool tipc_data_input(struct tipc_link *l, struct sk_buff *skb,
			    struct sk_buff_head *inputq)
{
	struct sk_buff_head *mc_inputq = l->bc_rcvlink->inputq;
	struct tipc_msg *hdr = buf_msg(skb);

	switch (msg_user(hdr)) {
	case TIPC_LOW_IMPORTANCE:
	case TIPC_MEDIUM_IMPORTANCE:
	case TIPC_HIGH_IMPORTANCE:
	case TIPC_CRITICAL_IMPORTANCE:
		if (unlikely(msg_in_group(hdr) || msg_mcast(hdr))) {
			skb_queue_tail(mc_inputq, skb);
			return true;
		}
		/* fall through */
	case CONN_MANAGER:
		skb_queue_tail(inputq, skb);
		return true;
	case GROUP_PROTOCOL:
		skb_queue_tail(mc_inputq, skb);
		return true;
	case NAME_DISTRIBUTOR:
		l->bc_rcvlink->state = LINK_ESTABLISHED;
		skb_queue_tail(l->namedq, skb);
		return true;
	case MSG_BUNDLER:
	case TUNNEL_PROTOCOL:
	case MSG_FRAGMENTER:
	case BCAST_PROTOCOL:
		return false;
	default:
		pr_warn("Dropping received illegal msg type\n");
		kfree_skb(skb);
		return true;
	};
}

static int (*klpe_tipc_link_input)(struct tipc_link *l, struct sk_buff *skb,
			   struct sk_buff_head *inputq,
			   struct sk_buff **reasm_skb);

static int klpr_tipc_link_tnl_rcv(struct tipc_link *l, struct sk_buff *skb,
			     struct sk_buff_head *inputq)
{
	struct sk_buff **reasm_skb = &l->failover_reasm_skb;
	struct sk_buff_head *fdefq = &l->failover_deferdq;
	struct tipc_msg *hdr = buf_msg(skb);
	struct sk_buff *iskb;
	int ipos = 0;
	int rc = 0;
	u16 seqno;

	/* SYNCH_MSG */
	if (msg_type(hdr) == SYNCH_MSG)
		goto drop;

	/* FAILOVER_MSG */
	if (!(*klpe_tipc_msg_extract)(skb, &iskb, &ipos)) {
		pr_warn_ratelimited("Cannot extract FAILOVER_MSG, defq: %d\n",
				    skb_queue_len(fdefq));
		return rc;
	}

	do {
		seqno = buf_seqno(iskb);

		if (unlikely(less(seqno, l->drop_point))) {
			kfree_skb(iskb);
			continue;
		}

		if (unlikely(seqno != l->drop_point)) {
			(*klpe___tipc_skb_queue_sorted)(fdefq, seqno, iskb);
			continue;
		}

		l->drop_point++;

		if (!tipc_data_input(l, iskb, inputq))
			rc |= (*klpe_tipc_link_input)(l, iskb, inputq, reasm_skb);
		if (unlikely(rc))
			break;
	} while ((iskb = __tipc_skb_dequeue(fdefq, l->drop_point)));

drop:
	kfree_skb(skb);
	return rc;
}

static bool tipc_link_release_pkts(struct tipc_link *l, u16 acked)
{
	bool released = false;
	struct sk_buff *skb, *tmp;

	skb_queue_walk_safe(&l->transmq, skb, tmp) {
		if (more(buf_seqno(skb), acked))
			break;
		__skb_unlink(skb, &l->transmq);
		kfree_skb(skb);
		released = true;
	}
	return released;
}

static int klpr_tipc_link_advance_transmq(struct tipc_link *l, u16 acked, u16 gap,
				     struct tipc_gap_ack_blks *ga,
				     struct sk_buff_head *xmitq)
{
	struct sk_buff *skb, *_skb, *tmp;
	struct tipc_msg *hdr;
	u16 bc_ack = l->bc_rcvlink->rcv_nxt - 1;
	u16 ack = l->rcv_nxt - 1;
	bool passed = false;
	u16 seqno, n = 0;
	int rc = 0;

	skb_queue_walk_safe(&l->transmq, skb, tmp) {
		seqno = buf_seqno(skb);

next_gap_ack:
		if (less_eq(seqno, acked)) {
			/* release skb */
			__skb_unlink(skb, &l->transmq);
			kfree_skb(skb);
		} else if (less_eq(seqno, acked + gap)) {
			/* First, check if repeated retrans failures occurs? */
			if (!passed && (*klpe_link_retransmit_failure)(l, l, &rc))
				return rc;
			passed = true;

			/* retransmit skb if unrestricted*/
			if (time_before(jiffies, TIPC_SKB_CB(skb)->nxt_retr))
				continue;
			TIPC_SKB_CB(skb)->nxt_retr = TIPC_UC_RETR_TIME;
			_skb = __pskb_copy(skb, LL_MAX_HEADER + MIN_H_SIZE,
					   GFP_ATOMIC);
			if (!_skb)
				continue;
			hdr = buf_msg(_skb);
			msg_set_ack(hdr, ack);
			msg_set_bcast_ack(hdr, bc_ack);
			_skb->priority = TC_PRIO_CONTROL;
			__skb_queue_tail(xmitq, _skb);
			l->stats.retransmitted++;

			/* Increase actual retrans counter & mark first time */
			if (!TIPC_SKB_CB(skb)->retr_cnt++)
				TIPC_SKB_CB(skb)->retr_stamp = jiffies;
		} else {
			/* retry with Gap ACK blocks if any */
			if (!ga || n >= ga->gack_cnt)
				break;
			acked = ntohs(ga->gacks[n].ack);
			gap = ntohs(ga->gacks[n].gap);
			n++;
			goto next_gap_ack;
		}
	}

	return 0;
}

static int klpr_tipc_link_build_nack_msg(struct tipc_link *l,
				    struct sk_buff_head *xmitq)
{
	u32 def_cnt = ++l->stats.deferred_recv;
	u32 defq_len = skb_queue_len(&l->deferdq);
	int match1, match2;

	if (link_is_bc_rcvlink(l)) {
		match1 = def_cnt & 0xf;
		match2 = klpr_tipc_own_addr(l->net) & 0xf;
		if (match1 == match2)
			return TIPC_LINK_SND_STATE;
		return 0;
	}

	if (defq_len >= 3 && !((defq_len - 3) % 16))
		(*klpe_tipc_link_build_proto_msg)(l, STATE_MSG, 0, 0, 0, 0, 0, xmitq);
	return 0;
}

int klpp_tipc_link_rcv(struct tipc_link *l, struct sk_buff *skb,
		  struct sk_buff_head *xmitq)
{
	struct sk_buff_head *defq = &l->deferdq;
	struct tipc_msg *hdr = buf_msg(skb);
	u16 seqno, rcv_nxt, win_lim;
	int rc = 0;

	/* Verify and update link state */
	if (unlikely(msg_user(hdr) == LINK_PROTOCOL))
		return klpp_tipc_link_proto_rcv(l, skb, xmitq);

	/* Don't send probe at next timeout expiration */
	l->silent_intv_cnt = 0;

	do {
		hdr = buf_msg(skb);
		seqno = msg_seqno(hdr);
		rcv_nxt = l->rcv_nxt;
		win_lim = rcv_nxt + TIPC_MAX_LINK_WIN;

		if (unlikely(!link_is_up(l))) {
			if (l->state == LINK_ESTABLISHING)
				rc = TIPC_LINK_UP_EVT;
			goto drop;
		}

		/* Drop if outside receive window */
		if (unlikely(less(seqno, rcv_nxt) || more(seqno, win_lim))) {
			l->stats.duplicates++;
			goto drop;
		}

		/* Forward queues and wake up waiting users */
		if (likely(tipc_link_release_pkts(l, msg_ack(hdr)))) {
			(*klpe_tipc_link_advance_backlog)(l, xmitq);
			if (unlikely(!skb_queue_empty(&l->wakeupq)))
				(*klpe_link_prepare_wakeup)(l);
		}

		/* Defer delivery if sequence gap */
		if (unlikely(seqno != rcv_nxt)) {
			(*klpe___tipc_skb_queue_sorted)(defq, seqno, skb);
			rc |= klpr_tipc_link_build_nack_msg(l, xmitq);
			break;
		}

		/* Deliver packet */
		l->rcv_nxt++;
		l->stats.recv_pkts++;

		if (unlikely(msg_user(hdr) == TUNNEL_PROTOCOL))
			rc |= klpr_tipc_link_tnl_rcv(l, skb, l->inputq);
		else if (!tipc_data_input(l, skb, l->inputq))
			rc |= (*klpe_tipc_link_input)(l, skb, l->inputq, &l->reasm_buf);
		if (unlikely(++l->rcv_unacked >= TIPC_MIN_LINK_WIN))
			rc |= (*klpe_tipc_link_build_state_msg)(l, xmitq);
		if (unlikely(rc & ~TIPC_LINK_SND_STATE))
			break;
	} while ((skb = __tipc_skb_dequeue(defq, l->rcv_nxt)));

	return rc;
drop:
	kfree_skb(skb);
	return rc;
}

static void (*klpe_tipc_link_build_proto_msg)(struct tipc_link *l, int mtyp, bool probe,
				      bool probe_reply, u16 rcvgap,
				      int tolerance, int priority,
				      struct sk_buff_head *xmitq);

static int klpp_tipc_link_proto_rcv(struct tipc_link *l, struct sk_buff *skb,
			       struct sk_buff_head *xmitq)
{
	struct tipc_msg *hdr = buf_msg(skb);
	struct tipc_gap_ack_blks *ga = NULL;
	u16 rcvgap = 0;
	u16 ack = msg_ack(hdr);
	u16 gap = msg_seq_gap(hdr);
	u16 peers_snd_nxt =  msg_next_sent(hdr);
	u16 peers_tol = msg_link_tolerance(hdr);
	u16 peers_prio = msg_linkprio(hdr);
	u16 rcv_nxt = l->rcv_nxt;
	/*
	 * Fix CVE-2022-0435
	 *  -1 line, +1 line
	 */
	u32 dlen = msg_data_sz(hdr);
	int mtyp = msg_type(hdr);
	bool reply = msg_probe(hdr);
	/*
	 * Fix CVE-2022-0435
	 *  -1 line, +1 line
	 */
	u32 glen = 0;
	void *data;
	char *if_name;
	int rc = 0;

	klpr_trace_tipc_proto_rcv(skb, false, l->name);

	/*
	 * Fix CVE-2022-0435
	 *  +2 lines
	 */
	if (dlen > U16_MAX)
		goto exit;

	if ((*klpe_tipc_link_is_blocked)(l) || !xmitq)
		goto exit;

	if (klpr_tipc_own_addr(l->net) > msg_prevnode(hdr))
		l->net_plane = msg_net_plane(hdr);

	skb_linearize(skb);
	hdr = buf_msg(skb);
	data = msg_data(hdr);

	if (!(*klpe_tipc_link_validate_msg)(l, hdr)) {
		klpr_trace_tipc_skb_dump(skb, false, "PROTO invalid (1)!");
		klpr_trace_tipc_link_dump(l, TIPC_DUMP_NONE, "PROTO invalid (1)!");
		goto exit;
	}

	switch (mtyp) {
	case RESET_MSG:
	case ACTIVATE_MSG:
		/* Complete own link name with peer's interface name */
		if_name =  strrchr(l->name, ':') + 1;
		if (sizeof(l->name) - (if_name - l->name) <= TIPC_MAX_IF_NAME)
			break;
		if (msg_data_sz(hdr) < TIPC_MAX_IF_NAME)
			break;
		strncpy(if_name, data, TIPC_MAX_IF_NAME);

		/* Update own tolerance if peer indicates a non-zero value */
		if (in_range(peers_tol, TIPC_MIN_LINK_TOL, TIPC_MAX_LINK_TOL)) {
			l->tolerance = peers_tol;
			l->bc_rcvlink->tolerance = peers_tol;
		}
		/* Update own priority if peer's priority is higher */
		if (in_range(peers_prio, l->priority + 1, TIPC_MAX_LINK_PRI))
			l->priority = peers_prio;

		/* If peer is going down we want full re-establish cycle */
		if (msg_peer_stopping(hdr)) {
			rc = (*klpe_tipc_link_fsm_evt)(l, LINK_FAILURE_EVT);
			break;
		}

		/* If this endpoint was re-created while peer was ESTABLISHING
		 * it doesn't know current session number. Force re-synch.
		 */
		if (mtyp == ACTIVATE_MSG && msg_dest_session_valid(hdr) &&
		    l->session != msg_dest_session(hdr)) {
			if (less(l->session, msg_dest_session(hdr)))
				l->session = msg_dest_session(hdr) + 1;
			break;
		}

		/* ACTIVATE_MSG serves as PEER_RESET if link is already down */
		if (mtyp == RESET_MSG || !link_is_up(l))
			rc = (*klpe_tipc_link_fsm_evt)(l, LINK_PEER_RESET_EVT);

		/* ACTIVATE_MSG takes up link if it was already locally reset */
		if (mtyp == ACTIVATE_MSG && l->state == LINK_ESTABLISHING)
			rc = TIPC_LINK_UP_EVT;

		l->peer_session = msg_session(hdr);
		l->in_session = true;
		l->peer_bearer_id = msg_bearer_id(hdr);
		if (l->mtu > msg_max_pkt(hdr))
			l->mtu = msg_max_pkt(hdr);
		break;

	case STATE_MSG:
		l->rcv_nxt_state = msg_seqno(hdr) + 1;

		/* Update own tolerance if peer indicates a non-zero value */
		if (in_range(peers_tol, TIPC_MIN_LINK_TOL, TIPC_MAX_LINK_TOL)) {
			l->tolerance = peers_tol;
			l->bc_rcvlink->tolerance = peers_tol;
		}
		/* Update own prio if peer indicates a different value */
		if ((peers_prio != l->priority) &&
		    in_range(peers_prio, 1, TIPC_MAX_LINK_PRI)) {
			l->priority = peers_prio;
			rc = (*klpe_tipc_link_fsm_evt)(l, LINK_FAILURE_EVT);
		}

		l->silent_intv_cnt = 0;
		l->stats.recv_states++;
		if (msg_probe(hdr))
			l->stats.recv_probes++;

		if (!link_is_up(l)) {
			if (l->state == LINK_ESTABLISHING)
				rc = TIPC_LINK_UP_EVT;
			break;
		}

		/* Receive Gap ACK blocks from peer if any */
		if (l->peer_caps & TIPC_GAP_ACK_BLOCK) {
			ga = (struct tipc_gap_ack_blks *)data;
			glen = ntohs(ga->len);
			/* sanity check: if failed, ignore Gap ACK blocks */
			if (glen != tipc_gap_ack_blks_sz(ga->gack_cnt))
				ga = NULL;
		}

		/*
		 * Fix CVE-2022-0435
		 *  +2 lines
		 */
		if(glen > dlen)
			break;
		klpp_tipc_mon_rcv(l->net, data + glen, dlen - glen, l->addr,
			     &l->mon_state, l->bearer_id);

		/* Send NACK if peer has sent pkts we haven't received yet */
		if (more(peers_snd_nxt, rcv_nxt) && !(*klpe_tipc_link_is_synching)(l))
			rcvgap = peers_snd_nxt - l->rcv_nxt;
		if (rcvgap || reply)
			(*klpe_tipc_link_build_proto_msg)(l, STATE_MSG, 0, reply,
						  rcvgap, 0, 0, xmitq);

		rc |= klpr_tipc_link_advance_transmq(l, ack, gap, ga, xmitq);

		/* If NACK, retransmit will now start at right position */
		if (gap)
			l->stats.recv_nacks++;

		(*klpe_tipc_link_advance_backlog)(l, xmitq);
		if (unlikely(!skb_queue_empty(&l->wakeupq)))
			(*klpe_link_prepare_wakeup)(l);
	}
exit:
	kfree_skb(skb);
	return rc;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1195308.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "tipc"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__tipc_skb_queue_sorted", (void *)&klpe___tipc_skb_queue_sorted,
	  "tipc" },
	{ "__tracepoint_tipc_link_dump",
	  (void *)&klpe___tracepoint_tipc_link_dump, "tipc" },
	{ "__tracepoint_tipc_proto_rcv",
	  (void *)&klpe___tracepoint_tipc_proto_rcv, "tipc" },
	{ "__tracepoint_tipc_skb_dump",
	  (void *)&klpe___tracepoint_tipc_skb_dump, "tipc" },
	{ "link_prepare_wakeup", (void *)&klpe_link_prepare_wakeup, "tipc" },
	{ "link_retransmit_failure", (void *)&klpe_link_retransmit_failure,
	  "tipc" },
	{ "tipc_link_advance_backlog", (void *)&klpe_tipc_link_advance_backlog,
	  "tipc" },
	{ "tipc_link_build_proto_msg", (void *)&klpe_tipc_link_build_proto_msg,
	  "tipc" },
	{ "tipc_link_build_state_msg", (void *)&klpe_tipc_link_build_state_msg,
	  "tipc" },
	{ "tipc_link_fsm_evt", (void *)&klpe_tipc_link_fsm_evt, "tipc" },
	{ "tipc_link_input", (void *)&klpe_tipc_link_input, "tipc" },
	{ "tipc_link_is_blocked", (void *)&klpe_tipc_link_is_blocked, "tipc" },
	{ "tipc_link_is_synching", (void *)&klpe_tipc_link_is_synching,
	  "tipc" },
	{ "tipc_link_validate_msg", (void *)&klpe_tipc_link_validate_msg,
	  "tipc" },
	{ "tipc_msg_extract", (void *)&klpe_tipc_msg_extract, "tipc" },
	{ "tipc_net_id", (void *)&klpe_tipc_net_id, "tipc" },
};

static int livepatch_bsc1195308_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1195308_module_nb = {
	.notifier_call = livepatch_bsc1195308_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1195308_link_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1195308_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1195308_link_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1195308_module_nb);
}
