/*
 * livepatch_bsc1191813
 *
 * Fix for CVE-2021-20322, bsc#1191813
 *
 *  Upstream commits:
 *  6457378fe796 ("ipv4: use siphash instead of Jenkins in fnhe_hashfun()")
 *  67d6d681e15b ("ipv4: make exception cache less predictible")
 *  4785305c05b2 ("ipv6: use siphash in rt6_exception_hash()")
 *  a00df2caffed ("ipv6: make exception cache less predictible")
 *
 *  SLE12-SP3 commits:
 *  c0cf71aeaaf9fca047fde14397a8e1b2dcfa5a5e
 *  3410ffcc51dd858fa6a4757e341e8ac165fea26f
 *
 *  SLE12-SP4, SLE12-SP5, SLE15 and SLE15-SP1 commits:
 *  238505e894d4ec80993fc49d3ec0c26ad34e52f0
 *  74af5bd03ad58489836a6a8269e219c34550fece
 *
 *  SLE15-SP2 and -SP3 commits:
 *  46555dae97faf328234ef9fd68d729281a896bd2
 *  191e9b31d6a1ee3303ac147e85c97ca3cb4ad596
 *  a7a1a7f8384d8b28875e21b6dfb64ef7984b4d93
 *  23f16ab0d5248479194556b343f7fd99686e14ab
 *
 *
 *  Copyright (c) 2021 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/livepatch.h>
#include <linux/vmalloc.h>
#include "livepatch_bsc1191813.h"
#include "bsc1191813_common.h"
#include "../shadow.h"

struct klp_bsc1191813_shared_state *klp_bsc1191813_shared_state;

#define KLP_BSC1191813_SHARED_STATE_ID KLP_SHADOW_ID(1191813, 0)

static int klp_bsc1191813_init_shared_state(void *obj,
					    void *shadow_data,
					    void *ctor_dat)
{
	struct klp_bsc1191813_shared_state *s = shadow_data;

	memset(s, 0, sizeof(*s));

	return 0;
}

static void __klp_bsc1191813_put_shared_state(void);

/* Must be called with module_mutex held. */
static int __klp_bsc1191813_get_shared_state(void)
{
	klp_bsc1191813_shared_state =
		klp_shadow_get_or_alloc(NULL, KLP_BSC1191813_SHARED_STATE_ID,
					sizeof(*klp_bsc1191813_shared_state),
					GFP_KERNEL,
					klp_bsc1191813_init_shared_state, NULL);
	if (!klp_bsc1191813_shared_state)
		return -ENOMEM;

	++klp_bsc1191813_shared_state->refcount;

	if (!klp_bsc1191813_shared_state->ipv4_eviction_history) {
		struct klp_bsc1191813_shared_state *s;

		s = klp_bsc1191813_shared_state;
		s->ipv4_eviction_history =
			vzalloc(sizeof(*s->ipv4_eviction_history));
		if (!s->ipv4_eviction_history) {
			__klp_bsc1191813_put_shared_state();
			return -ENOMEM;
		}

		s->ipv6_eviction_history =
			vzalloc(sizeof(*s->ipv6_eviction_history));
		if (!s->ipv6_eviction_history) {
			__klp_bsc1191813_put_shared_state();
			return -ENOMEM;
		}
	}

	return 0;
}

/* Must be called with module_mutex held. */
static void __klp_bsc1191813_put_shared_state(void)
{
	--klp_bsc1191813_shared_state->refcount;

	if (!klp_bsc1191813_shared_state->refcount) {
		vfree(klp_bsc1191813_shared_state->ipv4_eviction_history);
		vfree(klp_bsc1191813_shared_state->ipv6_eviction_history);
		klp_shadow_free(NULL, KLP_BSC1191813_SHARED_STATE_ID, NULL);
	}

	klp_bsc1191813_shared_state = NULL;
}


int livepatch_bsc1191813_init(void)
{	int ret;

	ret = livepatch_bsc1191813_ipv4_route_init();
	if (ret)
		return ret;

	ret = livepatch_bsc1191813_ipv6_route_init();
	if (ret) {
		livepatch_bsc1191813_ipv4_route_cleanup();
		return ret;
	}

	mutex_lock(&module_mutex);
	ret = __klp_bsc1191813_get_shared_state();
	mutex_unlock(&module_mutex);
	if (ret) {
		livepatch_bsc1191813_ipv6_route_cleanup();
		livepatch_bsc1191813_ipv4_route_cleanup();
		return ret;
	}

	return 0;
}

void livepatch_bsc1191813_cleanup(void)
{
	mutex_lock(&module_mutex);
	__klp_bsc1191813_put_shared_state();
	mutex_unlock(&module_mutex);
	livepatch_bsc1191813_ipv6_route_cleanup();
	livepatch_bsc1191813_ipv4_route_cleanup();
}
