/*
 * livepatch_bsc1191813
 *
 * Fix for CVE-2021-20322, bsc#1191813 (net/ipv6/route.c part)
 *
 *
 *  Copyright (c) 2021 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/ipv6/route.c */
#define pr_fmt(fmt) "IPv6: " fmt

#include <linux/capability.h>
#include <linux/errno.h>
#include <linux/export.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/route.h>
#include <linux/netdevice.h>
#include <linux/in6.h>
#include <linux/mroute6.h>
#include <linux/init.h>
#include <linux/if_arp.h>
#include <linux/seq_file.h>
#include <linux/nsproxy.h>
#include <linux/slab.h>
#include <linux/jhash.h>
#include <net/net_namespace.h>
#include <net/snmp.h>
#include <net/ipv6.h>
#include <net/ip6_fib.h>

/* klp-ccp: from include/net/ip6_fib.h */
static void (*klpe_fib6_update_sernum)(struct net *net, struct fib6_info *rt);

/* klp-ccp: from net/ipv6/route.c */
#include <net/ip6_route.h>

/* klp-ccp: from include/net/ip6_route.h */
static void (*klpe_fib6_force_start_gc)(struct net *net);

/* klp-ccp: from net/ipv6/route.c */
#include <net/ndisc.h>
#include <net/addrconf.h>
#include <linux/rtnetlink.h>
#include <net/netlink.h>
#include <net/lwtunnel.h>
#include <net/ip_tunnels.h>
#include <net/l3mdev.h>
#include <net/ip.h>
#include <linux/uaccess.h>
#include <linux/sysctl.h>
#include <linux/in6.h>
#include <net/flow.h>
#include <net/ip6_fib.h>
#include <linux/in6.h>
#include <net/flow.h>
#include <net/ip6_fib.h>
#include <linux/tracepoint.h>

/*
 * Live patching specific approach to mitigate against collision group
 * discovery: we cannot safely change the hash function on a running
 * system, but we can make finding collision groups harder. A vital
 * part of finding collision groups is to correlate evictions with
 * insertions: an attacker would reinsert some recently evicted entry
 * to see what falls out next and would have found a collision. The
 * livepatch mitigation is to break this chain by simply detect
 * reinsertions and have nothing new (or something unrelated) evicted
 * in this case. In order to implement this, maintain one ringbuffer
 * of recently evicted entries' addresses for each exception hash
 * bucket.
 */
#include "bsc1191813_common.h"

/* New. */
static void __klpp_ipv6_eviction_history_expire(struct klpp_ipv6_eviction_history *h)
{
	unsigned int i;

	for (i = h->head; i != h->tail;
	     i = (i + 1) % ARRAY_SIZE(h->entries)) {
		if (!time_after(jiffies, h->entries[i].expires))
			break;
	}

	h->head = i;
}

/* New. */
static bool __klpp_ipv6_eviction_history_lookup(struct klpp_ipv6_eviction_history *h,
						struct in6_addr *daddr)
{
	unsigned int i;

	__klpp_ipv6_eviction_history_expire(h);
	for (i = h->head; i != h->tail;
	     i = (i + 1) % ARRAY_SIZE(h->entries)) {
		if (ipv6_addr_equal(&h->entries[i].daddr, daddr))
			return true;
	}

	return false;
}

/* New. */
static void __klpp_ipv6_eviction_history_record(struct klpp_ipv6_eviction_history *h,
						struct in6_addr *daddr)
{
	unsigned int i;

	__klpp_ipv6_eviction_history_expire(h);

	i = h->tail;
	h->tail = (i + 1) % ARRAY_SIZE(h->entries);
	if (h->tail == h->head) {
		/* Ring buffer overrun. */
		h->head = (h->head + 1) % ARRAY_SIZE(h->entries);
	}

	h->entries[i].daddr = *daddr;
	h->entries[i].expires = jiffies + 300 * HZ;
}

/* New. */
static bool klpp_ipv6_eviction_history_lookup(u32 hval, struct in6_addr *daddr)
{
	struct klpp_ipv6_eviction_history *h;

	h = &(*klp_bsc1191813_shared_state->ipv6_eviction_history)[hval];
	return __klpp_ipv6_eviction_history_lookup(h, daddr);
}

/* New. */
static void klpp_ipv6_eviction_history_record(u32 hval, struct in6_addr *daddr)
{
	struct klpp_ipv6_eviction_history *h;

	h = &(*klp_bsc1191813_shared_state->ipv6_eviction_history)[hval];
	return __klpp_ipv6_eviction_history_record(h, daddr);
}


static spinlock_t (*klpe_rt6_exception_lock);

static void rt6_remove_exception(struct rt6_exception_bucket *bucket,
				 struct rt6_exception *rt6_ex)
{
	struct fib6_info *from;
	struct net *net;

	if (!bucket || !rt6_ex)
		return;

	net = dev_net(rt6_ex->rt6i->dst.dev);
	net->ipv6.rt6_stats->fib_rt_cache--;

	/* purge completely the exception to allow releasing the held resources:
	 * some [sk] cache may keep the dst around for unlimited time
	 */
	from = xchg((__force struct fib6_info **)&rt6_ex->rt6i->from, NULL);
	fib6_info_release(from);
	dst_dev_put(&rt6_ex->rt6i->dst);

	hlist_del_rcu(&rt6_ex->hlist);
	dst_release(&rt6_ex->rt6i->dst);
	kfree_rcu(rt6_ex, rcu);
	WARN_ON_ONCE(!bucket->depth);
	bucket->depth--;
}

/*
 * Fix CVE-2021-20322
 *  -1 line, +2 lines
 */
static void klpp_rt6_exception_remove_oldest(struct rt6_exception_bucket *rt6i_exception_bucket,
					     u32 hval)
{
	/*
	 * Fix CVE-2021-20322
	 *  +1 line
	 */
	struct rt6_exception_bucket *bucket;
	struct rt6_exception *rt6_ex, *oldest = NULL;

	/*
	 * Fix CVE-2021-20322
	 *  -2 lines, +3 lines
	 */
	if (!rt6i_exception_bucket || hval >= FIB6_EXCEPTION_BUCKET_SIZE)
		return;
	bucket = rt6i_exception_bucket + hval;

	hlist_for_each_entry(rt6_ex, &bucket->chain, hlist) {
		if (!oldest || time_before(rt6_ex->stamp, oldest->stamp))
			oldest = rt6_ex;
	}
	/*
	 * Fix CVE-2021-20322
	 *  +3 lines
	 */
	if (!oldest)
		return;

	klpp_ipv6_eviction_history_record(hval, &oldest->rt6i->rt6i_dst.addr);
	rt6_remove_exception(bucket, oldest);
}

static struct rt6_exception *
(*klpe___rt6_find_exception_spinlock)(struct rt6_exception_bucket **bucket,
			      const struct in6_addr *daddr,
			      const struct in6_addr *saddr);

static unsigned int fib6_mtu(const struct fib6_result *res)
{
	const struct fib6_nh *nh = res->nh;
	unsigned int mtu;

	if (res->f6i->fib6_pmtu) {
		mtu = res->f6i->fib6_pmtu;
	} else {
		struct net_device *dev = nh->fib_nh_dev;
		struct inet6_dev *idev;

		rcu_read_lock();
		idev = __in6_dev_get(dev);
		mtu = idev->cnf.mtu6;
		rcu_read_unlock();
	}

	mtu = min_t(unsigned int, mtu, IP6_MAX_MTU);

	return mtu - lwtunnel_headroom(nh->fib_nh_lws, mtu);
}

#define FIB6_EXCEPTION_BUCKET_FLUSHED  0x1UL

static bool fib6_nh_excptn_bucket_flushed(struct rt6_exception_bucket *bucket)
{
	unsigned long p = (unsigned long)bucket;

	return !!(p & FIB6_EXCEPTION_BUCKET_FLUSHED);
}

int klpp_rt6_insert_exception(struct rt6_info *nrt,
				const struct fib6_result *res)
{
	struct net *net = dev_net(nrt->dst.dev);
	/*
	 * Fix CVE-2021-20322
	 *  +1 line
	 */
	struct rt6_exception_bucket *rt6i_exception_bucket;
	struct rt6_exception_bucket *bucket;
	struct fib6_info *f6i = res->f6i;
	struct in6_addr *src_key = NULL;
	struct rt6_exception *rt6_ex;
	struct fib6_nh *nh = res->nh;
	/*
	 * Fix CVE-2021-20322
	 *  +2 lines
	 */
	int max_depth;
	int err = 0;

	spin_lock_bh(&(*klpe_rt6_exception_lock));

	/*
	 * Fix CVE-2021-20322
	 *  -14 lines, +15 lines
	 */
	rt6i_exception_bucket = rcu_dereference_protected(nh->rt6i_exception_bucket,
					  lockdep_is_held(&rt6_exception_lock));
	if (!rt6i_exception_bucket) {
		rt6i_exception_bucket = kcalloc(FIB6_EXCEPTION_BUCKET_SIZE, sizeof(*rt6i_exception_bucket),
				 GFP_ATOMIC);
		if (!rt6i_exception_bucket) {
			err = -ENOMEM;
			goto out;
		}
		rcu_assign_pointer(nh->rt6i_exception_bucket, rt6i_exception_bucket);
	} else if (fib6_nh_excptn_bucket_flushed(rt6i_exception_bucket)) {
		err = -EINVAL;
		goto out;
	}
	bucket = rt6i_exception_bucket;

#ifdef CONFIG_IPV6_SUBTREES
	if (f6i->fib6_src.plen)
		src_key = &nrt->rt6i_src.addr;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	if (dst_metric_raw(&nrt->dst, RTAX_MTU) >= fib6_mtu(res)) {
		err = -EINVAL;
		goto out;
	}

	rt6_ex = (*klpe___rt6_find_exception_spinlock)(&bucket, &nrt->rt6i_dst.addr,
					       src_key);
	if (rt6_ex)
		rt6_remove_exception(bucket, rt6_ex);

	rt6_ex = kzalloc(sizeof(*rt6_ex), GFP_ATOMIC);
	if (!rt6_ex) {
		err = -ENOMEM;
		goto out;
	}
	rt6_ex->rt6i = nrt;
	rt6_ex->stamp = jiffies;
	hlist_add_head_rcu(&rt6_ex->hlist, &bucket->chain);
	bucket->depth++;
	net->ipv6.rt6_stats->fib_rt_cache++;

	/*
	 * Fix CVE-2021-20322
	 *  -2 lines, +27 lines
	 */
	/* Randomize max depth to avoid some side channels attacks. */
	max_depth = FIB6_MAX_DEPTH + prandom_u32_max(FIB6_MAX_DEPTH);
	if (bucket->depth > max_depth) {
		u32 hval;

		hval = bucket - rt6i_exception_bucket;
		if (!klpp_ipv6_eviction_history_lookup(hval, &nrt->rt6i_dst.addr)) {
			while (bucket->depth > max_depth) {
				klpp_rt6_exception_remove_oldest(rt6i_exception_bucket,
								 hval);
			}
		} else {
			/*
			 * An entry for this destination
			 * address has been evicted recently,
			 * which is suspicious. Chances are
			 * an attacker is trying to reinsert
			 * it to find new collisions falling
			 * out from the bucket below. Pick
			 * some random bucket to evict from in
			 * order to not reveal any information
			 * on which pairs of entries collided.
			 */
			klpp_rt6_exception_remove_oldest(rt6i_exception_bucket,
							 prandom_u32_max(FIB6_EXCEPTION_BUCKET_SIZE));
		}
	}

out:
	spin_unlock_bh(&(*klpe_rt6_exception_lock));

	/* Update fn->fn_sernum to invalidate all cached dst */
	if (!err) {
		spin_lock_bh(&f6i->fib6_table->tb6_lock);
		(*klpe_fib6_update_sernum)(net, f6i);
		spin_unlock_bh(&f6i->fib6_table->tb6_lock);
		(*klpe_fib6_force_start_gc)(net);
	}

	return err;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/vermagic.h>
#include "livepatch_bsc1191813.h"
#include "bsc1191813_common.h"
#include "../kallsyms_relocs.h"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "__rt6_find_exception_spinlock",
	  (void *)&klpe___rt6_find_exception_spinlock },
	{ "fib6_force_start_gc", (void *)&klpe_fib6_force_start_gc },
	{ "fib6_update_sernum", (void *)&klpe_fib6_update_sernum },
	{ "rt6_exception_lock", (void *)&klpe_rt6_exception_lock },
};

int livepatch_bsc1191813_ipv6_route_init(void)
{
	return __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

void livepatch_bsc1191813_ipv6_route_cleanup(void)
{}
