/*
 * livepatch_bsc1191813
 *
 * Fix for CVE-2021-20322, bsc#1191813 (net/ipv4/route.c part)
 *
 *
 *  Copyright (c) 2021 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

/* klp-ccp: from net/ipv4/route.c */
#define pr_fmt(fmt) "IPv4: " fmt

#include <linux/module.h>
#include <linux/uaccess.h>
#include <linux/bitops.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/errno.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/init.h>
#include <linux/skbuff.h>
#include <linux/inetdevice.h>
#include <linux/pkt_sched.h>
#include <linux/mroute.h>
#include <linux/random.h>
#include <linux/rcupdate.h>
#include <linux/slab.h>
#include <linux/jhash.h>
#include <net/dst.h>
#include <net/dst_metadata.h>
#include <net/net_namespace.h>
#include <net/route.h>
#include <net/inetpeer.h>
#include <net/sock.h>
#include <net/ip_fib.h>
#include <net/nexthop.h>
#include <net/arp.h>
#include <net/lwtunnel.h>
#include <net/rtnetlink.h>
#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

#include <net/ip_tunnels.h>
#include <net/l3mdev.h>
/* klp-ccp: from net/ipv4/fib_lookup.h */
#include <linux/types.h>
#include <linux/list.h>
#include <net/ip_fib.h>
#include <net/nexthop.h>

/* klp-ccp: from net/ipv4/route.c */
/*
 * Live patching specific approach to mitigate against collision group
 * discovery: we cannot safely change the hash function on a running
 * system, but we can make finding collision groups harder. A vital
 * part of finding collision groups is to correlate evictions with
 * insertions: an attacker would reinsert some recently evicted entry
 * to see what falls out next and would have found a collision. The
 * livepatch mitigation is to break this chain by simply detect
 * reinsertions and have nothing new (or something unrelated) evicted
 * in this case. In order to implement this, maintain one ringbuffer
 * of recently evicted entries' addresses for each exception hash
 * bucket.
 */
#include "bsc1191813_common.h"

/* New. */
static void __klpp_ipv4_eviction_history_expire(struct klpp_ipv4_eviction_history *h)
{
	unsigned int i;

	for (i = h->head; i != h->tail;
	     i = (i + 1) % ARRAY_SIZE(h->entries)) {
		if (!time_after(jiffies, h->entries[i].expires))
			break;
	}

	h->head = i;
}

/* New. */
static bool __klpp_ipv4_eviction_history_lookup(struct klpp_ipv4_eviction_history *h,
						__be32 daddr)
{
	unsigned int i;

	__klpp_ipv4_eviction_history_expire(h);
	for (i = h->head; i != h->tail;
	     i = (i + 1) % ARRAY_SIZE(h->entries)) {
		if (h->entries[i].daddr == daddr)
			return true;
	}

	return false;
}

/* New. */
static void __klpp_ipv4_eviction_history_record(struct klpp_ipv4_eviction_history *h,
						__be32 daddr)
{
	unsigned int i;

	__klpp_ipv4_eviction_history_expire(h);

	i = h->tail;
	h->tail = (i + 1) % ARRAY_SIZE(h->entries);
	if (h->tail == h->head) {
		/* Ring buffer overrun. */
		h->head = (h->head + 1) % ARRAY_SIZE(h->entries);
	}

	h->entries[i].daddr = daddr;
	h->entries[i].expires = jiffies + 300 * HZ;
}

/* New. */
static bool klpp_ipv4_eviction_history_lookup(u32 hval, __be32 daddr)
{
	struct klpp_ipv4_eviction_history *h;

	h = &(*klp_bsc1191813_shared_state->ipv4_eviction_history)[hval];
	return __klpp_ipv4_eviction_history_lookup(h, daddr);
}

/* New. */
static void klpp_ipv4_eviction_history_record(u32 hval, __be32 daddr)
{
	struct klpp_ipv4_eviction_history *h;

	h = &(*klp_bsc1191813_shared_state->ipv4_eviction_history)[hval];
	return __klpp_ipv4_eviction_history_record(h, daddr);
}


static spinlock_t (*klpe_fnhe_lock);

static void (*klpe_fnhe_flush_routes)(struct fib_nh_exception *fnhe);

/* New. */
static void klpp_fnhe_remove_oldest(struct fnhe_hash_bucket *nhc_exceptions,
				    u32 hval)
{
	struct fnhe_hash_bucket *hash = nhc_exceptions + hval;
	struct fib_nh_exception __rcu **fnhe_p, **oldest_p;
	struct fib_nh_exception *fnhe, *oldest = NULL;

	for (fnhe_p = &hash->chain; ; fnhe_p = &fnhe->fnhe_next) {
		fnhe = rcu_dereference_protected(*fnhe_p,
						 lockdep_is_held(&fnhe_lock));
		if (!fnhe)
			break;
		if (!oldest ||
		    time_before(fnhe->fnhe_stamp, oldest->fnhe_stamp)) {
			oldest = fnhe;
			oldest_p = fnhe_p;
		}
	}
	if (!oldest)
		return;

	klpp_ipv4_eviction_history_record(hval, oldest->fnhe_daddr);
	(*klpe_fnhe_flush_routes)(oldest);
	*oldest_p = oldest->fnhe_next;
	kfree_rcu(oldest, rcu);
}

static u32 (*klpe_fnhe_hashfun_local_fnhe_hashrnd);
static bool (*klpe_fnhe_hashfun_local____done);

static void klp_fnhe_hashfun_fnhe_hashrnd_init_once(void)
{
	/*
	 * Livepatch adjustment: simulate the DO_ONCE() from
	 * fnhe_hashfun()'s net_get_random_once() invocation. The
	 * function here would get called only from DO_ONCE() itself,
	 * and thus the global once_lock is held.
	 */
	if (!(*klpe_fnhe_hashfun_local____done)) {
		get_random_bytes(&(*klpe_fnhe_hashfun_local_fnhe_hashrnd),
				 sizeof(*klpe_fnhe_hashfun_local_fnhe_hashrnd));
		*klpe_fnhe_hashfun_local____done = true;
	}
}

static inline u32 klpp_fnhe_hashfun(__be32 daddr)
{
	u32 hval;

	DO_ONCE(klp_fnhe_hashfun_fnhe_hashrnd_init_once);
	hval = jhash_1word((__force u32) daddr, (*klpe_fnhe_hashfun_local_fnhe_hashrnd));
	return hash_32(hval, FNHE_HASH_SHIFT);
}

static void fill_route_from_fnhe(struct rtable *rt, struct fib_nh_exception *fnhe)
{
	rt->rt_pmtu = fnhe->fnhe_pmtu;
	rt->rt_mtu_locked = fnhe->fnhe_mtu_locked;
	rt->dst.expires = fnhe->fnhe_expires;

	if (fnhe->fnhe_gw) {
		rt->rt_flags |= RTCF_REDIRECTED;
		rt->rt_uses_gateway = 1;
		rt->rt_gw_family = AF_INET;
		rt->rt_gw4 = fnhe->fnhe_gw;
	}
}

void klpp_update_or_create_fnhe(struct fib_nh_common *nhc, __be32 daddr,
				  __be32 gw, u32 pmtu, bool lock,
				  unsigned long expires)
{
	/*
	 * Fix CVE-2021-20322
	 *  +1 line
	 */
	struct fnhe_hash_bucket *nhc_exceptions;
	struct fnhe_hash_bucket *hash;
	struct fib_nh_exception *fnhe;
	struct rtable *rt;
	u32 genid, hval;
	unsigned int i;
	int depth;

	genid = fnhe_genid(dev_net(nhc->nhc_dev));
	hval = klpp_fnhe_hashfun(daddr);

	spin_lock_bh(&(*klpe_fnhe_lock));

	/*
	 * Fix CVE-2021-20322
	 *  -7 lines, +8 lines
	 */
	nhc_exceptions = rcu_dereference(nhc->nhc_exceptions);
	if (!nhc_exceptions) {
		nhc_exceptions = kcalloc(FNHE_HASH_SIZE, sizeof(*nhc_exceptions), GFP_ATOMIC);
		if (!nhc_exceptions)
			goto out_unlock;
		rcu_assign_pointer(nhc->nhc_exceptions, nhc_exceptions);
	}
	hash = nhc_exceptions;

	hash += hval;

	depth = 0;
	for (fnhe = rcu_dereference(hash->chain); fnhe;
	     fnhe = rcu_dereference(fnhe->fnhe_next)) {
		if (fnhe->fnhe_daddr == daddr)
			break;
		depth++;
	}

	if (fnhe) {
		if (fnhe->fnhe_genid != genid)
			fnhe->fnhe_genid = genid;
		if (gw)
			fnhe->fnhe_gw = gw;
		if (pmtu) {
			fnhe->fnhe_pmtu = pmtu;
			fnhe->fnhe_mtu_locked = lock;
		}
		fnhe->fnhe_expires = max(1UL, expires);
		/* Update all cached dsts too */
		rt = rcu_dereference(fnhe->fnhe_rth_input);
		if (rt)
			fill_route_from_fnhe(rt, fnhe);
		rt = rcu_dereference(fnhe->fnhe_rth_output);
		if (rt)
			fill_route_from_fnhe(rt, fnhe);
	} else {
		/*
		 * Fix CVE-2021-20322
		 *  -10 lines, +33 lines
		 */
		/* Randomize max depth to avoid some side channels attacks. */
		int max_depth = FNHE_RECLAIM_DEPTH +
				prandom_u32_max(FNHE_RECLAIM_DEPTH);

		if (depth > max_depth) {
			if (!klpp_ipv4_eviction_history_lookup(hval, daddr)) {
				while (depth > max_depth) {
					klpp_fnhe_remove_oldest(nhc_exceptions, hval);
					depth--;
				}
			} else {
				/*
				 * The entry for this destination
				 * address has been evicted recently,
				 * which is suspicious. Chances are
				 * an attacker is trying to reinsert
				 * it to find new collisions falling
				 * out from the bucket below. Pick
				 * some random bucket to evict from in
				 * order to not reveal any information
				 * on which pairs of entries collided.
				 */
				klpp_fnhe_remove_oldest(nhc_exceptions,
							prandom_u32_max(FNHE_HASH_SIZE));
			}
		}

		fnhe = kzalloc(sizeof(*fnhe), GFP_ATOMIC);
		if (!fnhe)
			goto out_unlock;

		fnhe->fnhe_next = hash->chain;

		fnhe->fnhe_genid = genid;
		fnhe->fnhe_daddr = daddr;
		fnhe->fnhe_gw = gw;
		fnhe->fnhe_pmtu = pmtu;
		fnhe->fnhe_mtu_locked = lock;
		fnhe->fnhe_expires = max(1UL, expires);

		/*
		 * Fix CVE-2021-20322
		 *  +1 line
		 */
		rcu_assign_pointer(hash->chain, fnhe);

		/* Exception created; mark the cached routes for the nexthop
		 * stale, so anyone caching it rechecks if this exception
		 * applies to them.
		 */
		rt = rcu_dereference(nhc->nhc_rth_input);
		if (rt)
			rt->dst.obsolete = DST_OBSOLETE_KILL;

		for_each_possible_cpu(i) {
			struct rtable __rcu **prt;
			prt = per_cpu_ptr(nhc->nhc_pcpu_rth_output, i);
			rt = rcu_dereference(*prt);
			if (rt)
				rt->dst.obsolete = DST_OBSOLETE_KILL;
		}
	}

	fnhe->fnhe_stamp = jiffies;

out_unlock:
	spin_unlock_bh(&(*klpe_fnhe_lock));
}



#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/vermagic.h>
#include "livepatch_bsc1191813.h"
#include "../kallsyms_relocs.h"

static int klp_resolve_fnhe_hashfun_locals(void)
{
	struct klp_kallsyms_reloc locals[] = {
		{ .addr = (void *)&klpe_fnhe_hashfun_local_fnhe_hashrnd },
		{ .addr = (void *)&klpe_fnhe_hashfun_local____done },
	};

#if IS_ENABLED(CONFIG_X86_64)
	if(!strcmp(UTS_RELEASE, "5.3.18-24.43-default")) { /* SLE15-SP2_Update_8 */
		locals[0].symname = "fnhe_hashrnd.81096";
		locals[1].symname = "___done.81099";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.46-default")) { /* SLE15-SP2_Update_9 */
		locals[0].symname = "fnhe_hashrnd.81101";
		locals[1].symname = "___done.81104";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.49-default")) { /* SLE15-SP2_Update_10 */
		locals[0].symname = "fnhe_hashrnd.81103";
		locals[1].symname = "___done.81106";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.52-default")) { /* SLE15-SP2_Update_11 */
		locals[0].symname = "fnhe_hashrnd.81112";
		locals[1].symname = "___done.81115";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.61-default")) { /* SLE15-SP2_Update_12 */
		locals[0].symname = "fnhe_hashrnd.81124";
		locals[1].symname = "___done.81127";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.64-default")) { /* SLE15-SP2_Update_13 */
		locals[0].symname = "fnhe_hashrnd.81124";
		locals[1].symname = "___done.81127";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.67-default")) { /* SLE15-SP2_Update_14 */
		locals[0].symname = "fnhe_hashrnd.81125";
		locals[1].symname = "___done.81128";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.53.4-default")) { /* SLE15-SP2_Update_15 */
		locals[0].symname = "fnhe_hashrnd.81112";
		locals[1].symname = "___done.81115";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.70-default")) { /* SLE15-SP2_Update_16 */
		locals[0].symname = "fnhe_hashrnd.81065";
		locals[1].symname = "___done.81068";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.75-default")) { /* SLE15-SP2_Update_17 */
		locals[0].symname = "fnhe_hashrnd.81067";
		locals[1].symname = "___done.81070";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.78-default")) { /* SLE15-SP2_Update_18 */
		locals[0].symname = "fnhe_hashrnd.81086";
		locals[1].symname = "___done.81089";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.83-default")) { /* SLE15-SP2_Update_19 */
		locals[0].symname = "fnhe_hashrnd.81122";
		locals[1].symname = "___done.81125";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.86-default")) { /* SLE15-SP2_Update_20 */
		locals[0].symname = "fnhe_hashrnd.81134";
		locals[1].symname = "___done.81137";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.93-default")) { /* SLE15-SP2_Update_21 */
		locals[0].symname = "fnhe_hashrnd.81146";
		locals[1].symname = "___done.81149";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-57-default")) { /* SLE15-SP3_Update_0 */
		locals[0].symname = "fnhe_hashrnd.83135";
		locals[1].symname = "___done.83138";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.5-default")) { /* SLE15-SP3_Update_1 */
		locals[0].symname = "fnhe_hashrnd.83152";
		locals[1].symname = "___done.83155";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.10-default")) { /* SLE15-SP3_Update_2 */
		locals[0].symname = "fnhe_hashrnd.83093";
		locals[1].symname = "___done.83096";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.13-default")) { /* SLE15-SP3_Update_3 */
		locals[0].symname = "fnhe_hashrnd.83093";
		locals[1].symname = "___done.83096";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.16-default")) { /* SLE15-SP3_Update_4 */
		locals[0].symname = "fnhe_hashrnd.83093";
		locals[1].symname = "___done.83096";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.19-default")) { /* SLE15-SP3_Update_5 */
		locals[0].symname = "fnhe_hashrnd.83092";
		locals[1].symname = "___done.83095";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.24-default")) { /* SLE15-SP3_Update_6 */
		locals[0].symname = "fnhe_hashrnd.83131";
		locals[1].symname = "___done.83134";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.27-default")) { /* SLE15-SP3_Update_7 */
		locals[0].symname = "fnhe_hashrnd.83139";
		locals[1].symname = "___done.83142";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.34-default")) { /* SLE15-SP3_Update_9 */
		locals[0].symname = "fnhe_hashrnd.83159";
		locals[1].symname = "___done.83162";

	}
#elif IS_ENABLED(CONFIG_PPC64)
	if(!strcmp(UTS_RELEASE, "5.3.18-24.43-default")) { /* SLE15-SP2_Update_8 */
		locals[0].symname = "fnhe_hashrnd.82516";
		locals[1].symname = "___done.82519";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.46-default")) { /* SLE15-SP2_Update_9 */
		locals[0].symname = "fnhe_hashrnd.82613";
		locals[1].symname = "___done.82616";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.49-default")) { /* SLE15-SP2_Update_10 */
		locals[0].symname = "fnhe_hashrnd.82665";
		locals[1].symname = "___done.82668";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.52-default")) { /* SLE15-SP2_Update_11 */
		locals[0].symname = "fnhe_hashrnd.82650";
		locals[1].symname = "___done.82653";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.61-default")) { /* SLE15-SP2_Update_12 */
		locals[0].symname = "fnhe_hashrnd.82658";
		locals[1].symname = "___done.82661";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.64-default")) { /* SLE15-SP2_Update_13 */
		locals[0].symname = "fnhe_hashrnd.82658";
		locals[1].symname = "___done.82661";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.67-default")) { /* SLE15-SP2_Update_14 */
		locals[0].symname = "fnhe_hashrnd.82659";
		locals[1].symname = "___done.82662";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.53.4-default")) { /* SLE15-SP2_Update_15 */
		locals[0].symname = "fnhe_hashrnd.82650";
		locals[1].symname = "___done.82653";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.70-default")) { /* SLE15-SP2_Update_16 */
		locals[0].symname = "fnhe_hashrnd.82597";
		locals[1].symname = "___done.82600";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.75-default")) { /* SLE15-SP2_Update_17 */
		locals[0].symname = "fnhe_hashrnd.82599";
		locals[1].symname = "___done.82602";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.78-default")) { /* SLE15-SP2_Update_18 */
		locals[0].symname = "fnhe_hashrnd.82618";
		locals[1].symname = "___done.82621";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.83-default")) { /* SLE15-SP2_Update_19 */
		locals[0].symname = "fnhe_hashrnd.82654";
		locals[1].symname = "___done.82657";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.86-default")) { /* SLE15-SP2_Update_20 */
		locals[0].symname = "fnhe_hashrnd.82672";
		locals[1].symname = "___done.82675";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.93-default")) { /* SLE15-SP2_Update_21 */
		locals[0].symname = "fnhe_hashrnd.82684";
		locals[1].symname = "___done.82687";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-57-default")) { /* SLE15-SP3_Update_0 */
		locals[0].symname = "fnhe_hashrnd.85076";
		locals[1].symname = "___done.85079";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.5-default")) { /* SLE15-SP3_Update_1 */
		locals[0].symname = "fnhe_hashrnd.85092";
		locals[1].symname = "___done.85095";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.10-default")) { /* SLE15-SP3_Update_2 */
		locals[0].symname = "fnhe_hashrnd.85030";
		locals[1].symname = "___done.85033";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.13-default")) { /* SLE15-SP3_Update_3 */
		locals[0].symname = "fnhe_hashrnd.85030";
		locals[1].symname = "___done.85033";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.16-default")) { /* SLE15-SP3_Update_4 */
		locals[0].symname = "fnhe_hashrnd.85030";
		locals[1].symname = "___done.85033";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.19-default")) { /* SLE15-SP3_Update_5 */
		locals[0].symname = "fnhe_hashrnd.85029";
		locals[1].symname = "___done.85032";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.24-default")) { /* SLE15-SP3_Update_6 */
		locals[0].symname = "fnhe_hashrnd.85075";
		locals[1].symname = "___done.85078";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.27-default")) { /* SLE15-SP3_Update_7 */
		locals[0].symname = "fnhe_hashrnd.85082";
		locals[1].symname = "___done.85085";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.34-default")) { /* SLE15-SP3_Update_9 */
		locals[0].symname = "fnhe_hashrnd.85107";
		locals[1].symname = "___done.85110";

	}
#elif IS_ENABLED(CONFIG_S390)
	if(!strcmp(UTS_RELEASE, "5.3.18-24.43-default")) { /* SLE15-SP2_Update_8 */
		locals[0].symname = "fnhe_hashrnd.77096";
		locals[1].symname = "___done.77099";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.46-default")) { /* SLE15-SP2_Update_9 */
		locals[0].symname = "fnhe_hashrnd.77193";
		locals[1].symname = "___done.77196";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.49-default")) { /* SLE15-SP2_Update_10 */
		locals[0].symname = "fnhe_hashrnd.77193";
		locals[1].symname = "___done.77196";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.52-default")) { /* SLE15-SP2_Update_11 */
		locals[0].symname = "fnhe_hashrnd.77199";
		locals[1].symname = "___done.77202";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.61-default")) { /* SLE15-SP2_Update_12 */
		locals[0].symname = "fnhe_hashrnd.77214";
		locals[1].symname = "___done.77217";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.64-default")) { /* SLE15-SP2_Update_13 */
		locals[0].symname = "fnhe_hashrnd.77214";
		locals[1].symname = "___done.77217";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.67-default")) { /* SLE15-SP2_Update_14 */
		locals[0].symname = "fnhe_hashrnd.77215";
		locals[1].symname = "___done.77218";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.53.4-default")) { /* SLE15-SP2_Update_15 */
		locals[0].symname = "fnhe_hashrnd.77199";
		locals[1].symname = "___done.77202";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.70-default")) { /* SLE15-SP2_Update_16 */
		locals[0].symname = "fnhe_hashrnd.77153";
		locals[1].symname = "___done.77156";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.75-default")) { /* SLE15-SP2_Update_17 */
		locals[0].symname = "fnhe_hashrnd.77155";
		locals[1].symname = "___done.77158";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.78-default")) { /* SLE15-SP2_Update_18 */
		locals[0].symname = "fnhe_hashrnd.77174";
		locals[1].symname = "___done.77177";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.83-default")) { /* SLE15-SP2_Update_19 */
		locals[0].symname = "fnhe_hashrnd.77210";
		locals[1].symname = "___done.77213";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.86-default")) { /* SLE15-SP2_Update_20 */
		locals[0].symname = "fnhe_hashrnd.77217";
		locals[1].symname = "___done.77220";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-24.93-default")) { /* SLE15-SP2_Update_21 */
		locals[0].symname = "fnhe_hashrnd.77229";
		locals[1].symname = "___done.77232";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-57-default")) { /* SLE15-SP3_Update_0 */
		locals[0].symname = "fnhe_hashrnd.80125";
		locals[1].symname = "___done.80128";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.5-default")) { /* SLE15-SP3_Update_1 */
		locals[0].symname = "fnhe_hashrnd.80140";
		locals[1].symname = "___done.80143";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.10-default")) { /* SLE15-SP3_Update_2 */
		locals[0].symname = "fnhe_hashrnd.80078";
		locals[1].symname = "___done.80081";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.13-default")) { /* SLE15-SP3_Update_3 */
		locals[0].symname = "fnhe_hashrnd.80078";
		locals[1].symname = "___done.80081";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.16-default")) { /* SLE15-SP3_Update_4 */
		locals[0].symname = "fnhe_hashrnd.80078";
		locals[1].symname = "___done.80081";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.19-default")) { /* SLE15-SP3_Update_5 */
		locals[0].symname = "fnhe_hashrnd.80077";
		locals[1].symname = "___done.80080";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.24-default")) { /* SLE15-SP3_Update_6 */
		locals[0].symname = "fnhe_hashrnd.80117";
		locals[1].symname = "___done.80120";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.27-default")) { /* SLE15-SP3_Update_7 */
		locals[0].symname = "fnhe_hashrnd.80124";
		locals[1].symname = "___done.80127";

	} else if(!strcmp(UTS_RELEASE, "5.3.18-59.34-default")) { /* SLE15-SP3_Update_9 */
		locals[0].symname = "fnhe_hashrnd.80144";
		locals[1].symname = "___done.80147";

	}
#else
#error "Architecture not supported by livepatch."
#endif
	else {
		WARN(1, "kernel version not supported by livepatch\n");
		return -ENOTSUPP;

	}

	return __klp_resolve_kallsyms_relocs(locals, ARRAY_SIZE(locals));
}

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "fnhe_flush_routes", (void *)&klpe_fnhe_flush_routes },
	{ "fnhe_lock", (void *)&klpe_fnhe_lock },
};

int livepatch_bsc1191813_ipv4_route_init(void)
{	int ret;

	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	if (ret)
		return ret;

	ret = klp_resolve_fnhe_hashfun_locals();
	if (ret)
		return ret;

	return 0;
}

void livepatch_bsc1191813_ipv4_route_cleanup(void)
{}
