/*
 * kgraft_patch_bsc1165631_geneve
 *
 * Fix for CVE-2020-1749, bsc#1165631 (geneve.ko part)
 * and CVE-2020-25645, bsc#1177513.
 *
 *  Copyright (c) 2020 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if !IS_MODULE(CONFIG_GENEVE)
#error "Live patch supports only CONFIG_GENEVE=m"
#endif

#define KGR_PATCHED_MODULE "geneve"

/* klp-ccp: from drivers/net/geneve.c */
#define pr_fmt(fmt) KGR_PATCHED_MODULE ": " fmt

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/hash.h>
#include <net/dst_metadata.h>
#include <net/gro_cells.h>
#include <net/rtnetlink.h>
#include <net/geneve.h>

#include "kgr_patch_bsc1165631.h"
#include "../kallsyms_relocs.h"

/* klp-ccp: from include/net/udp_tunnel.h */
static int (*kgre_udp_tunnel_xmit_skb)(struct rtable *rt, struct sock *sk, struct sk_buff *skb,
			__be32 src, __be32 dst, __u8 tos, __u8 ttl,
			__be16 df, __be16 src_port, __be16 dst_port,
			bool xnet, bool nocheck);

#if IS_ENABLED(CONFIG_IPV6)
static int (*kgre_udp_tunnel6_xmit_skb)(struct dst_entry *dst, struct sock *sk,
			 struct sk_buff *skb,
			 struct net_device *dev, struct in6_addr *saddr,
			 struct in6_addr *daddr,
			 __u8 prio, __u8 ttl, __be16 src_port,
			 __be16 dst_port, bool nocheck);
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

/* klp-ccp: from drivers/net/geneve.c */
#define VNI_HASH_BITS		10
#define VNI_HASH_SIZE		(1<<VNI_HASH_BITS)

union geneve_addr {
	struct sockaddr_in sin;
	struct sockaddr_in6 sin6;
	struct sockaddr sa;
};

struct geneve_dev {
	struct hlist_node  hlist;	/* vni hash table */
	struct net	   *net;	/* netns for packet i/o */
	struct net_device  *dev;	/* netdev for geneve tunnel */
	struct geneve_sock *sock4;	/* IPv4 socket used for geneve tunnel */
#if IS_ENABLED(CONFIG_IPV6)
	struct geneve_sock *sock6;	/* IPv6 socket used for geneve tunnel */
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	u8                 vni[3];	/* virtual network ID for tunnel */
	u8                 ttl;		/* TTL override */
	u8                 tos;		/* TOS override */
	union geneve_addr  remote;	/* IP address for link partner */
	struct list_head   next;	/* geneve's per namespace list */
	__be16		   dst_port;
	bool		   collect_md;
	struct gro_cells   gro_cells;
};

struct geneve_sock {
	bool			collect_md;
	struct list_head	list;
	struct socket		*sock;
	struct rcu_head		rcu;
	int			refcnt;
	struct udp_offload	udp_offloads;
	struct hlist_head	vni_list[VNI_HASH_SIZE];
};

static int (*kgre_geneve_build_skb)(struct rtable *rt, struct sk_buff *skb,
			    __be16 tun_flags, u8 vni[3], u8 opt_len, u8 *opt,
			    bool csum, bool xnet);

#if IS_ENABLED(CONFIG_IPV6)
static int (*kgre_geneve6_build_skb)(struct dst_entry *dst, struct sk_buff *skb,
			     __be16 tun_flags, u8 vni[3], u8 opt_len, u8 *opt,
			     bool csum, bool xnet);

#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

static struct rtable *kgrp_geneve_get_v4_rt(struct sk_buff *skb,
				       struct net_device *dev,
				       struct flowi4 *fl4,
					/*
					 * Fix CVE-2020-25645
					 *  -1 line, +2 lines
					 */
				       struct ip_tunnel_info *info,
					__be16 dport, __be16 sport)
{
	struct geneve_dev *geneve = netdev_priv(dev);
	struct rtable *rt = NULL;
	__u8 tos;

	memset(fl4, 0, sizeof(*fl4));
	fl4->flowi4_mark = skb->mark;
	fl4->flowi4_proto = IPPROTO_UDP;
	/*
	 * Fix CVE-2020-25645
	 *  +2 lines
	 */
	fl4->fl4_dport = dport;
	fl4->fl4_sport = sport;

	if (info) {
		fl4->daddr = info->key.u.ipv4.dst;
		fl4->saddr = info->key.u.ipv4.src;
		fl4->flowi4_tos = RT_TOS(info->key.tos);
	} else {
		tos = geneve->tos;
		if (tos == 1) {
			const struct iphdr *iip = ip_hdr(skb);

			tos = ip_tunnel_get_dsfield(iip, skb);
		}

		fl4->flowi4_tos = RT_TOS(tos);
		fl4->daddr = geneve->remote.sin.sin_addr.s_addr;
	}

	rt = ip_route_output_key(geneve->net, fl4);
	if (IS_ERR(rt)) {
		netdev_dbg(dev, "no route to %pI4\n", &fl4->daddr);
		return ERR_PTR(-ENETUNREACH);
	}
	if (rt->dst.dev == dev) { /* is this necessary? */
		netdev_dbg(dev, "circular route to %pI4\n", &fl4->daddr);
		ip_rt_put(rt);
		return ERR_PTR(-ELOOP);
	}
	return rt;
}

#if IS_ENABLED(CONFIG_IPV6)
static struct dst_entry *kgrp_geneve_get_v6_dst(struct sk_buff *skb,
					   struct net_device *dev,
					   struct flowi6 *fl6,
					   /*
					    * Fix CVE-2020-25645
					    *  -1 line, +2 lines
					    */
					   struct ip_tunnel_info *info,
					   __be16 dport, __be16 sport)
{
	struct geneve_dev *geneve = netdev_priv(dev);
	struct geneve_sock *gs6 = geneve->sock6;
	struct dst_entry *dst = NULL;
	__u8 prio;

	memset(fl6, 0, sizeof(*fl6));
	fl6->flowi6_mark = skb->mark;
	fl6->flowi6_proto = IPPROTO_UDP;
	/*
	 * Fix CVE-2020-25645
	 *  +2 lines
	 */
	fl6->fl6_dport = dport;
	fl6->fl6_sport = sport;

	if (info) {
		fl6->daddr = info->key.u.ipv6.dst;
		fl6->saddr = info->key.u.ipv6.src;
		fl6->flowlabel = ip6_make_flowinfo(RT_TOS(info->key.tos), 0);
	} else {
		prio = geneve->tos;
		if (prio == 1) {
			const struct iphdr *iip = ip_hdr(skb);

			prio = ip_tunnel_get_dsfield(iip, skb);
		}

		fl6->flowlabel = ip6_make_flowinfo(RT_TOS(prio), 0);
		fl6->daddr = geneve->remote.sin6.sin6_addr;
	}

	/*
	 * Fix CVE-2020-1749
	 *  -1 line, +3 lines
	 */
	dst = kgrp_ip6_dst_lookup_flow(geneve->net, gs6->sock->sk, fl6,
				       NULL);
	if (IS_ERR(dst)) {
		netdev_dbg(dev, "no route to %pI6\n", &fl6->daddr);
		return ERR_PTR(-ENETUNREACH);
	}
	if (dst->dev == dev) { /* is this necessary? */
		netdev_dbg(dev, "circular route to %pI6\n", &fl6->daddr);
		dst_release(dst);
		return ERR_PTR(-ELOOP);
	}

	return dst;
}
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

static void tunnel_id_to_vni(__be64 tun_id, __u8 *vni)
{
#ifdef __BIG_ENDIAN
#error "klp-ccp: non-taken branch"
#else
	vni[0] = (__force __u8)((__force u64)tun_id >> 40);
	vni[1] = (__force __u8)((__force u64)tun_id >> 48);
	vni[2] = (__force __u8)((__force u64)tun_id >> 56);
#endif
}

static netdev_tx_t kgrp_geneve_xmit_skb(struct sk_buff *skb, struct net_device *dev,
				   struct ip_tunnel_info *info)
{
	struct geneve_dev *geneve = netdev_priv(dev);
	struct geneve_sock *gs4 = geneve->sock4;
	struct rtable *rt = NULL;
	int err = -EINVAL;
	struct flowi4 fl4;
	__u8 tos, ttl;
	__be16 sport;
	bool udp_csum;
	__be16 df;
	bool xnet = !net_eq(geneve->net, dev_net(geneve->dev));

	if (geneve->collect_md) {
		if (unlikely(!info || !(info->mode & IP_TUNNEL_INFO_TX))) {
			netdev_dbg(dev, "no tunnel metadata\n");
			goto tx_error;
		}
		if (info && ip_tunnel_info_af(info) != AF_INET)
			goto tx_error;
	}

	/*
	 * Fix CVE-2020-25645
	 *  -1 line, +3 lines
	 */
	sport = udp_flow_src_port(geneve->net, skb, 1, USHRT_MAX, true);
	rt = kgrp_geneve_get_v4_rt(skb, dev, &fl4, info,
				   geneve->dst_port, sport);
	if (IS_ERR(rt)) {
		err = PTR_ERR(rt);
		goto tx_error;
	}

	/*
	 * Fix CVE-2020-25645
	 *  -1 line
	 */
	skb_reset_mac_header(skb);

	if (info) {
		const struct ip_tunnel_key *key = &info->key;
		u8 *opts = NULL;
		u8 vni[3];

		tunnel_id_to_vni(key->tun_id, vni);
		if (key->tun_flags & TUNNEL_GENEVE_OPT)
			opts = ip_tunnel_info_opts(info);

		udp_csum = !!(key->tun_flags & TUNNEL_CSUM);
		err = (*kgre_geneve_build_skb)(rt, skb, key->tun_flags, vni,
				       info->options_len, opts, udp_csum, xnet);
		if (unlikely(err))
			goto err;

		tos = ip_tunnel_ecn_encap(key->tos, ip_hdr(skb), skb);
		ttl = key->ttl;
		df = key->tun_flags & TUNNEL_DONT_FRAGMENT ? htons(IP_DF) : 0;
	} else {
		udp_csum = false;
		err = (*kgre_geneve_build_skb)(rt, skb, 0, geneve->vni,
				       0, NULL, udp_csum, xnet);
		if (unlikely(err))
			goto err;

		tos = ip_tunnel_ecn_encap(fl4.flowi4_tos, ip_hdr(skb), skb);
		ttl = geneve->ttl;
		if (!ttl && IN_MULTICAST(ntohl(fl4.daddr)))
			ttl = 1;
		ttl = ttl ? : ip4_dst_hoplimit(&rt->dst);
		df = 0;
	}
	err = (*kgre_udp_tunnel_xmit_skb)(rt, gs4->sock->sk, skb, fl4.saddr, fl4.daddr,
				  tos, ttl, df, sport, geneve->dst_port,
				  !net_eq(geneve->net, dev_net(geneve->dev)),
				  !udp_csum);

	iptunnel_xmit_stats(err, &dev->stats, dev->tstats);
	return NETDEV_TX_OK;

tx_error:
	dev_kfree_skb(skb);
err:
	if (err == -ELOOP)
		dev->stats.collisions++;
	else if (err == -ENETUNREACH)
		dev->stats.tx_carrier_errors++;
	else
		dev->stats.tx_errors++;
	return NETDEV_TX_OK;
}

#if IS_ENABLED(CONFIG_IPV6)
static netdev_tx_t kgrp_geneve6_xmit_skb(struct sk_buff *skb, struct net_device *dev,
				    struct ip_tunnel_info *info)
{
	struct geneve_dev *geneve = netdev_priv(dev);
	struct geneve_sock *gs6 = geneve->sock6;
	struct dst_entry *dst = NULL;
	int err = -EINVAL;
	struct flowi6 fl6;
	__u8 prio, ttl;
	__be16 sport;
	bool udp_csum;
	bool xnet = !net_eq(geneve->net, dev_net(geneve->dev));

	if (geneve->collect_md) {
		if (unlikely(!info || !(info->mode & IP_TUNNEL_INFO_TX))) {
			netdev_dbg(dev, "no tunnel metadata\n");
			goto tx_error;
		}
	}

	/*
	 * Fix CVE-2020-25645
	 *  -1 line, +3 lines
	 */
	sport = udp_flow_src_port(geneve->net, skb, 1, USHRT_MAX, true);
	dst = kgrp_geneve_get_v6_dst(skb, dev, &fl6, info,
				     geneve->dst_port, sport);
	if (IS_ERR(dst)) {
		err = PTR_ERR(dst);
		goto tx_error;
	}

	/*
	 * Fix CVE-2020-25645
	 *  -1 line
	 */
	skb_reset_mac_header(skb);

	if (info) {
		const struct ip_tunnel_key *key = &info->key;
		u8 *opts = NULL;
		u8 vni[3];

		tunnel_id_to_vni(key->tun_id, vni);
		if (key->tun_flags & TUNNEL_GENEVE_OPT)
			opts = ip_tunnel_info_opts(info);

		udp_csum = !!(key->tun_flags & TUNNEL_CSUM);
		err = (*kgre_geneve6_build_skb)(dst, skb, key->tun_flags, vni,
					info->options_len, opts,
					udp_csum, xnet);
		if (unlikely(err))
			goto err;

		prio = ip_tunnel_ecn_encap(key->tos, ip_hdr(skb), skb);
		ttl = key->ttl;
	} else {
		udp_csum = false;
		err = (*kgre_geneve6_build_skb)(dst, skb, 0, geneve->vni,
					0, NULL, udp_csum, xnet);
		if (unlikely(err))
			goto err;

		prio = ip_tunnel_ecn_encap(ip6_tclass(fl6.flowlabel),
					   ip_hdr(skb), skb);
		ttl = geneve->ttl;
		if (!ttl && ipv6_addr_is_multicast(&fl6.daddr))
			ttl = 1;
		ttl = ttl ? : ip6_dst_hoplimit(dst);
	}
	err = (*kgre_udp_tunnel6_xmit_skb)(dst, gs6->sock->sk, skb, dev,
				   &fl6.saddr, &fl6.daddr, prio, ttl,
				   sport, geneve->dst_port, !udp_csum);
	return NETDEV_TX_OK;

tx_error:
	dev_kfree_skb(skb);
err:
	if (err == -ELOOP)
		dev->stats.collisions++;
	else if (err == -ENETUNREACH)
		dev->stats.tx_carrier_errors++;
	else
		dev->stats.tx_errors++;
	return NETDEV_TX_OK;
}
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif

netdev_tx_t kgrp_geneve_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct geneve_dev *geneve = netdev_priv(dev);
	struct ip_tunnel_info *info = NULL;

	if (geneve->collect_md)
		info = skb_tunnel_info(skb);

#if IS_ENABLED(CONFIG_IPV6)
	if ((info && ip_tunnel_info_af(info) == AF_INET6) ||
	    (!info && geneve->remote.sa.sa_family == AF_INET6))
		return kgrp_geneve6_xmit_skb(skb, dev, info);
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	return kgrp_geneve_xmit_skb(skb, dev, info);
}

int kgrp_geneve_fill_metadata_dst(struct net_device *dev, struct sk_buff *skb)
{
	struct ip_tunnel_info *info = skb_tunnel_info(skb);
	struct geneve_dev *geneve = netdev_priv(dev);
	struct rtable *rt;
	struct flowi4 fl4;
#if IS_ENABLED(CONFIG_IPV6)
	struct dst_entry *dst;
	struct flowi6 fl6;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	/*
	 * Fix CVE-2020-25645
	 *  +1 line
	 */
	__be16 sport;

	if (ip_tunnel_info_af(info) == AF_INET) {
		/*
		 * Fix CVE-2020-25645
		 *  -1 line, +4 lines
		 */
		sport = udp_flow_src_port(geneve->net, skb,
					  1, USHRT_MAX, true);
		rt = kgrp_geneve_get_v4_rt(skb, dev, &fl4, info,
					   geneve->dst_port, sport);
		if (IS_ERR(rt))
			return PTR_ERR(rt);

		ip_rt_put(rt);
		info->key.u.ipv4.src = fl4.saddr;
#if IS_ENABLED(CONFIG_IPV6)
	} else if (ip_tunnel_info_af(info) == AF_INET6) {
		/*
		 * Fix CVE-2020-25645
		 *  -1 line, +4 lines
		 */
		sport = udp_flow_src_port(geneve->net, skb,
					  1, USHRT_MAX, true);
		dst = kgrp_geneve_get_v6_dst(skb, dev, &fl6, info,
					     geneve->dst_port, sport);
		if (IS_ERR(dst))
			return PTR_ERR(dst);

		dst_release(dst);
		info->key.u.ipv6.src = fl6.saddr;
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	} else {
		return -EINVAL;
	}

	/*
	 * Fix CVE-2020-25645
	 *  -2 lines, +1 line
	 */
	info->key.tp_src = sport;
	info->key.tp_dst = geneve->dst_port;
	return 0;
}



static struct kgr_kallsyms_reloc kgr_funcs[] = {
	{ "udp_tunnel_xmit_skb", (void *)&kgre_udp_tunnel_xmit_skb,
	  "udp_tunnel" },
	{ "udp_tunnel6_xmit_skb", (void *)&kgre_udp_tunnel6_xmit_skb,
	  "ip6_udp_tunnel" },
	{ "geneve_build_skb", (void *)&kgre_geneve_build_skb, "geneve" },
	{ "geneve6_build_skb", (void *)&kgre_geneve6_build_skb, "geneve" },
};

static int kgr_patch_bsc1165631_geneve_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	ret = __kgr_resolve_kallsyms_relocs(kgr_funcs, ARRAY_SIZE(kgr_funcs));
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1165631_geneve_module_nb = {
	.notifier_call = kgr_patch_bsc1165631_geneve_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1165631_geneve_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(KGR_PATCHED_MODULE)) {
		ret = __kgr_resolve_kallsyms_relocs(kgr_funcs,
						    ARRAY_SIZE(kgr_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1165631_geneve_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1165631_geneve_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1165631_geneve_module_nb);
}
