/*
 * livepatch_bsc1194737
 *
 * Fix for CVE-2022-0185, bsc#1194737
 *
 *  Upstream commit:
 *  722d94847de2 ("vfs: fs_context: fix up param length parsing in
 *                 legacy_parse_param")
 *
 *  SLE12-SP3 commit:
 *  not affected
 *
 *  SLE12-SP4, SLE12-SP5, SLE15 and SLE15-SP1 commit:
 *  not affected
 *
 *  SLE15-SP2 and -SP3 commit:
 *  5ee58343d5f8f3f18158f11639785daa91dedd80
 *
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


/* klp-ccp: from fs/fs_context.c */
#define pr_fmt(fmt) "fs_conext" ": " fmt

#include <linux/module.h>
#include <linux/fs_context.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/magic.h>
#include <linux/security.h>

/* klp-ccp: from fs/mount.h */
#include <linux/mount.h>
#include <linux/seq_file.h>
#include <linux/ns_common.h>

/* klp-ccp: from fs/fs_context.c */
enum legacy_fs_param {
	LEGACY_FS_UNSET_PARAMS,
	LEGACY_FS_MONOLITHIC_PARAMS,
	LEGACY_FS_INDIVIDUAL_PARAMS,
};

struct legacy_fs_context {
	char			*legacy_data;	/* Data page for legacy filesystems */
	size_t			data_size;
	enum legacy_fs_param	param_type;
};

/* Livepatching specific */
#define KLPP_FC_FOR_LOG(fc) ((fc && fc->log && fc->log->owner) ? fc : NULL)

int klpp_legacy_parse_param(struct fs_context *fc, struct fs_parameter *param)
{
	struct legacy_fs_context *ctx = fc->fs_private;
	unsigned int size = ctx->data_size;
	size_t len = 0;

	if (strcmp(param->key, "source") == 0) {
		if (param->type != fs_value_is_string)
			return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Non-string source");
		if (fc->source)
			return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Multiple sources");
		fc->source = param->string;
		param->string = NULL;
		return 0;
	}

	if ((fc->fs_type->fs_flags & FS_HAS_SUBTYPE) &&
	    strcmp(param->key, "subtype") == 0) {
		if (param->type != fs_value_is_string)
			return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Non-string subtype");
		if (fc->subtype)
			return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Multiple subtype");
		fc->subtype = param->string;
		param->string = NULL;
		return 0;
	}

	if (ctx->param_type == LEGACY_FS_MONOLITHIC_PARAMS)
		return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Can't mix monolithic and individual options");

	switch (param->type) {
	case fs_value_is_string:
		len = 1 + param->size;
		/* Fall through */
	case fs_value_is_flag:
		len += strlen(param->key);
		break;
	default:
		return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Parameter type for '%s' not supported",
			      param->key);
	}

	/*
	 * Fix CVE-2022-0185
	 *  -1 line, +1 line
	 */
	if (size + len + 2 > PAGE_SIZE)
		return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Cumulative options too large");
	if (strchr(param->key, ',') ||
	    (param->type == fs_value_is_string &&
	     memchr(param->string, ',', param->size)))
		return invalf(KLPP_FC_FOR_LOG(fc), "VFS: Legacy: Option '%s' contained comma",
			      param->key);
	if (!ctx->legacy_data) {
		ctx->legacy_data = kmalloc(PAGE_SIZE, GFP_KERNEL);
		if (!ctx->legacy_data)
			return -ENOMEM;
	}

	ctx->legacy_data[size++] = ',';
	len = strlen(param->key);
	memcpy(ctx->legacy_data + size, param->key, len);
	size += len;
	if (param->type == fs_value_is_string) {
		ctx->legacy_data[size++] = '=';
		memcpy(ctx->legacy_data + size, param->string, param->size);
		size += param->size;
	}
	ctx->legacy_data[size] = '\0';
	ctx->data_size = size;
	ctx->param_type = LEGACY_FS_INDIVIDUAL_PARAMS;
	return 0;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1194737.h"
