/*
 * kgraft_patch_bsc1133191_memory
 *
 * Fix for CVE-2019-11487, bsc#1133191 (mm/memory.c part)
 *
 *  Copyright (c) 2019 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/swap.h>
#include <linux/swapops.h>
#include <linux/hugetlb.h>
#include <linux/pagemap.h>
#include <linux/rmap.h>
#include <asm/cacheflush.h>
#include <asm/mmu_context.h>
#include "kgr_patch_bsc1133191_memory.h"
#include "kgr_patch_bsc1133191_mm.h"
#include "kallsyms_relocs.h"


#if !IS_ENABLED(CONFIG_MIGRATION)
#error "Live patch suppports only CONFIG_MIGRATION=y"
#endif

#if !IS_ENABLED(CONFIG_ARCH_WANT_GENERAL_HUGETLB)
#error "Live patch suppports only CONFIG_ARCH_WANT_GENERAL_HUGETLB=y"
#endif

#if !defined(__HAVE_ARCH_GATE_AREA)
#error "Expected __HAVE_ARCH_GATE_AREA on x86_64"
#endif


static struct mm_struct *kgr_init_mm;

static void (*kgr_migration_entry_wait)(struct mm_struct *mm, pmd_t *pmd,
					unsigned long address);
static struct page *(*kgr_vm_normal_page)(struct vm_area_struct *vma,
					  unsigned long addr, pte_t pte);
static void (*kgr_lru_add_drain)(void);
static void (*kgr_mlock_vma_page)(struct page *page);
static struct vm_area_struct *(*kgr_find_extend_vma)(struct mm_struct * mm,
						     unsigned long addr);
static int (*kgr_in_gate_area)(struct mm_struct *mm, unsigned long addr);
static struct vm_area_struct *(*kgr_get_gate_vma)(struct mm_struct *mm);
static int (*kgr_handle_mm_fault)(struct mm_struct *mm,
				  struct vm_area_struct *vma,
				  unsigned long address, unsigned int flags);

#ifdef CONFIG_TRANSPARENT_HUGEPAGE
static struct page *(*kgr_follow_huge_addr)(struct mm_struct *mm,
					    unsigned long address,
					    int write);
static struct page *(*kgr_follow_huge_pud)(struct mm_struct *mm,
					   unsigned long address,
					   pud_t *pud, int flags);
static struct page * (*kgr_follow_huge_pmd)(struct mm_struct *mm,
					    unsigned long address,
					    pmd_t *pmd, int flags);
static void (*kgr__split_huge_page_pmd)(struct vm_area_struct *vma,
					unsigned long address,
					pmd_t *pmd);
static struct page *(*kgr_follow_trans_huge_pmd)(struct vm_area_struct *vma,
						 unsigned long addr,
						 pmd_t *pmd,
						 unsigned int flags);
#else /* CONFIG_TRANSPARENT_HUGEPAGE */
#define kgr_follow_huge_addr follow_huge_addr
#define kgr_follow_huge_pud follow_huge_pud
#define kgr_follow_huge_pmd follow_huge_pmd
#define kgr_follow_trans_huge_pmd follow_trans_huge_pmd
#endif /* CONFIG_TRANSPARENT_HUGEPAGE */

#ifdef CONFIG_HUGETLB_PAGE
static int (*kgr_pmd_huge)(pmd_t pmd);
static int (*kgr_pud_huge)(pud_t pud);
#else /* CONFIG_HUGETLB_PAGE */
#define kgr_pmd_huge pmd_huge
#define kgr_pud_huge pud_huge
#endif /* CONFIG_HUGETLB_PAGE */

#ifdef CONFIG_HUGETLBFS
static pte_t *(*kgr_huge_pte_offset)(struct mm_struct *mm, unsigned long addr);
static int (*kgr_hugetlb_fault)(struct mm_struct *mm,
				struct vm_area_struct *vma,
				unsigned long address, unsigned int flags);
#endif /* CONFIG_HUGETLBFS */

static struct kgr_kallsyms_reloc kgr_funcs[] = {
	{ "init_mm", (void *)&kgr_init_mm },
	{ "migration_entry_wait", (void *)&kgr_migration_entry_wait },
	{ "vm_normal_page", (void *)&kgr_vm_normal_page },
	{ "lru_add_drain", (void *)&kgr_lru_add_drain },
	{ "mlock_vma_page", (void *)&kgr_mlock_vma_page },
	{ "find_extend_vma", (void *)&kgr_find_extend_vma },
	{ "in_gate_area", (void *)&kgr_in_gate_area },
	{ "get_gate_vma", (void *)&kgr_get_gate_vma },
	{ "handle_mm_fault", (void *)&kgr_handle_mm_fault },
#ifdef CONFIG_TRANSPARENT_HUGEPAGE
	{ "follow_huge_addr", (void *)&kgr_follow_huge_addr },
	{ "follow_huge_pud", (void *)&kgr_follow_huge_pud },
	{ "follow_huge_pmd", (void *)&kgr_follow_huge_pmd },
	{ "__split_huge_page_pmd", (void *)&kgr__split_huge_page_pmd },
	{ "follow_trans_huge_pmd", (void *)&kgr_follow_trans_huge_pmd },
#endif
#ifdef CONFIG_HUGETLB_PAGE
	{ "pmd_huge", (void *)&kgr_pmd_huge },
	{ "pud_huge", (void *)&kgr_pud_huge },
#endif
#ifdef CONFIG_HUGETLBFS
	{ "huge_pte_offset", (void *)&kgr_huge_pte_offset },
	{ "hugetlb_fault", (void *)&kgr_hugetlb_fault },
#endif
};


/* from arch/x86/include/asm/pgtable.h */
/* resolve to non-EXPORTed init_mm */
#define kgr_pgd_offset_k(address) pgd_offset(&(*kgr_init_mm), (address))

/* from include/linux/huge_mm.h */
#ifdef CONFIG_TRANSPARENT_HUGEPAGE
/* resolve to non-EXPORTed __split_huge_page_pmd() */
#define kgr_split_huge_page_pmd(__vma, __address, __pmd)		\
	do {								\
		pmd_t *____pmd = (__pmd);				\
		if (unlikely(pmd_trans_huge(*____pmd)))		\
			kgr__split_huge_page_pmd(__vma, __address,	\
					____pmd);			\
	}  while (0)
#else
#define kgr_split_huge_page_pmd split_huge_page_pmd
#endif


/* from mm/internal.h */
/* inlined */
static inline void kgr__get_page_tail_foll(struct page *page,
					   bool get_page_head)
{
	/*
	 * If we're getting a tail page, the elevated page->_count is
	 * required only in the head page and we will elevate the head
	 * page->_count and tail page->_mapcount.
	 *
	 * We elevate page_tail->_mapcount for tail pages to force
	 * page_tail->_count to be zero at all times to avoid getting
	 * false positives from get_page_unless_zero() with
	 * speculative page access (like in
	 * page_cache_get_speculative()) on tail pages.
	 */
	VM_BUG_ON(atomic_read(&page->first_page->_count) <= 0);
	VM_BUG_ON(atomic_read(&page->_count) != 0);
	VM_BUG_ON(page_mapcount(page) < 0);
	if (get_page_head)
		atomic_inc(&page->first_page->_count);
	if (compound_tail_refcounted(page->first_page))
		atomic_inc(&page->_mapcount);
}

#ifdef CONFIG_HUGETLBFS
/* inlined */
static inline struct page *kgr_mem_map_offset(struct page *base, int offset)
{
	if (unlikely(offset >= MAX_ORDER_NR_PAGES))
		return pfn_to_page(page_to_pfn(base) + offset);
	return base + offset;
}
#endif


/* from mm/memory.c */
/* inlined */
static inline bool kgr_can_follow_write_pte(pte_t pte, unsigned int flags)
{
	return pte_write(pte) ||
		((flags & FOLL_FORCE) && (flags & FOLL_COW) && pte_dirty(pte));
}


/* from mm/hugetlb.c */
#ifdef CONFIG_HUGETLBFS
/* inlined */
static pgoff_t kgr_vma_hugecache_offset(struct hstate *h,
			struct vm_area_struct *vma, unsigned long address)
{
	return ((address - vma->vm_start) >> huge_page_shift(h)) +
			(vma->vm_pgoff >> huge_page_order(h));
}

/* inlined */
static bool kgr_hugetlbfs_pagecache_present(struct hstate *h,
			struct vm_area_struct *vma, unsigned long address)
{
	struct address_space *mapping;
	pgoff_t idx;
	struct page *page;

	mapping = vma->vm_file->f_mapping;
	idx = kgr_vma_hugecache_offset(h, vma, address);

	page = find_get_page(mapping, idx);
	if (page)
		put_page(page);
	return page != NULL;
}
#endif


/* patched, inlined */
/*
 * Fix CVE-2019-11487
 *  -1 line, +1 line
 */
static inline bool kgr_get_page_foll(struct page *page, bool check)
{
	/*
	 * Fix CVE-2019-11487
	 *  -1 line, +1 line
	 */
	if (unlikely(PageTail(page))) {
		/*
		 * This is safe only because
		 * __split_huge_page_refcount() can't run under
		 * get_page_foll() because we hold the proper PT lock.
		 */
		/*
		 * Fix CVE-2019-11487
		 *  +3 lines
		 */
		if (check && WARN_ON_ONCE(
				kgr_page_ref_count(compound_head(page)) <= 0))
			return false;
		kgr__get_page_tail_foll(page, true);
	/*
	 * Fix CVE-2019-11487
	 *  -1 line, +1 line
	 */
	} else {
		/*
		 * Getting a normal page or the head of a compound page
		 * requires to already have an elevated page->_count.
		 */
		VM_BUG_ON(atomic_read(&page->_count) <= 0);
		/*
		 * Fix CVE-2019-11487
		 *  +2 lines
		 */
		if (check && WARN_ON_ONCE(kgr_page_ref_count(page) <= 0))
			return false;
		atomic_inc(&page->_count);
	}
	/*
	 * Fix CVE-2019-11487
	 *  +1 line
	 */
	return true;
}

/* patched */
struct page *kgr_follow_page_mask(struct vm_area_struct *vma,
				  unsigned long address, unsigned int flags,
				  unsigned int *page_mask)
{
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pte_t *ptep, pte;
	spinlock_t *ptl;
	struct page *page;
	struct mm_struct *mm = vma->vm_mm;

	*page_mask = 0;

	page = kgr_follow_huge_addr(mm, address, flags & FOLL_WRITE);
	if (!IS_ERR(page)) {
		BUG_ON(flags & FOLL_GET);
		goto out;
	}

	page = NULL;
	pgd = pgd_offset(mm, address);
	if (pgd_none(*pgd) || unlikely(pgd_bad(*pgd)))
		goto no_page_table;

	pud = pud_offset(pgd, address);
	if (pud_none(*pud))
		goto no_page_table;
	if (kgr_pud_huge(*pud) && vma->vm_flags & VM_HUGETLB) {
		page = kgr_follow_huge_pud(mm, address, pud, flags);
		if (page)
			goto out;
		goto no_page_table;
	}
	if (unlikely(pud_bad(*pud)))
		goto no_page_table;

	pmd = pmd_offset(pud, address);
	if (pmd_none(*pmd))
		goto no_page_table;
	if (kgr_pmd_huge(*pmd) && vma->vm_flags & VM_HUGETLB) {
		page = kgr_follow_huge_pmd(mm, address, pmd, flags);
		if (page)
			goto out;
		goto no_page_table;
	}
	if ((flags & FOLL_NUMA) && pmd_protnone(*pmd))
		goto no_page_table;
	if (pmd_trans_huge(*pmd)) {
		if (flags & FOLL_SPLIT) {
			kgr_split_huge_page_pmd(vma, address, pmd);
			goto split_fallthrough;
		}
		ptl = pmd_lock(mm, pmd);
		if (likely(pmd_trans_huge(*pmd))) {
			if (unlikely(pmd_trans_splitting(*pmd))) {
				spin_unlock(ptl);
				wait_split_huge_page(vma->anon_vma, pmd);
			} else {
				page = kgr_follow_trans_huge_pmd(vma, address,
								 pmd, flags);
				spin_unlock(ptl);
				*page_mask = HPAGE_PMD_NR - 1;
				goto out;
			}
		} else
			spin_unlock(ptl);
		/* fall through */
	}
split_fallthrough:
	if (unlikely(pmd_bad(*pmd)))
		goto no_page_table;

	ptep = pte_offset_map_lock(mm, pmd, address, &ptl);

	pte = *ptep;
	if (!pte_present(pte)) {
		swp_entry_t entry;
		/*
		 * KSM's break_ksm() relies upon recognizing a ksm page
		 * even while it is being migrated, so for that case we
		 * need migration_entry_wait().
		 */
		if (likely(!(flags & FOLL_MIGRATION)))
			goto no_page;
		if (pte_none(pte) || pte_file(pte))
			goto no_page;
		entry = pte_to_swp_entry(pte);
		if (!is_migration_entry(entry))
			goto no_page;
		pte_unmap_unlock(ptep, ptl);
		kgr_migration_entry_wait(mm, pmd, address);
		goto split_fallthrough;
	}
	if ((flags & FOLL_NUMA) && pte_protnone(pte))
		goto no_page;
	if ((flags & FOLL_WRITE) && !kgr_can_follow_write_pte(pte, flags))
		goto unlock;

	page = kgr_vm_normal_page(vma, address, pte);
	if (unlikely(!page)) {
		if ((flags & FOLL_DUMP) ||
		    !is_zero_pfn(pte_pfn(pte)))
			goto bad_page;
		page = pte_page(pte);
	}

	/*
	 * Fix CVE-2019-11487
	 *  -2 lines, +6 lines
	 */
	if (flags & FOLL_GET) {
		if (!kgr_get_page_foll(page, true)) {
			page = ERR_PTR(-ENOMEM);
			goto unlock;
		}
	}
	if (flags & FOLL_TOUCH) {
		if ((flags & FOLL_WRITE) &&
		    !pte_dirty(pte) && !PageDirty(page))
			set_page_dirty(page);
		/*
		 * pte_mkyoung() would be more correct here, but atomic care
		 * is needed to avoid losing the dirty bit: it is easier to use
		 * mark_page_accessed().
		 */
		mark_page_accessed(page);
	}
	if ((flags & FOLL_MLOCK) && (vma->vm_flags & VM_LOCKED)) {
		/*
		 * The preliminary mapping check is mainly to avoid the
		 * pointless overhead of lock_page on the ZERO_PAGE
		 * which might bounce very badly if there is contention.
		 *
		 * If the page is already locked, we don't need to
		 * handle it now - vmscan will handle it later if and
		 * when it attempts to reclaim the page.
		 */
		if (page->mapping && trylock_page(page)) {
			kgr_lru_add_drain();  /* push cached pages to LRU */
			/*
			 * Because we lock page here, and migration is
			 * blocked by the pte's page reference, and we
			 * know the page is still mapped, we don't even
			 * need to check for file-cache page truncation.
			 */
			kgr_mlock_vma_page(page);
			unlock_page(page);
		}
	}
unlock:
	pte_unmap_unlock(ptep, ptl);
out:
	return page;

bad_page:
	pte_unmap_unlock(ptep, ptl);
	return ERR_PTR(-EFAULT);

no_page:
	pte_unmap_unlock(ptep, ptl);
	if (!pte_none(pte))
		return page;

no_page_table:
	/*
	 * When core dumping an enormous anonymous area that nobody
	 * has touched so far, we don't want to allocate unnecessary pages or
	 * page tables.  Return error instead of NULL to skip handle_mm_fault,
	 * then get_dump_page() will return NULL to leave a hole in the dump.
	 * But we can only make this optimization where a hole would surely
	 * be zero-filled if handle_mm_fault() actually did handle it.
	 */
	if ((flags & FOLL_DUMP) &&
	    (!vma->vm_ops || !vma->vm_ops->fault))
		return ERR_PTR(-EFAULT);
	return page;
}

#ifdef CONFIG_HUGETLBFS
/*
 * patched, not inlined, but the only caller, __get_user_pages(), gets
 * also patched
 */
static long kgr_follow_hugetlb_page(struct mm_struct *mm,
				    struct vm_area_struct *vma,
				    struct page **pages,
				    struct vm_area_struct **vmas,
				    unsigned long *position,
				    unsigned long *nr_pages,
				    long i, unsigned int flags)
{
	unsigned long pfn_offset;
	unsigned long vaddr = *position;
	unsigned long remainder = *nr_pages;
	struct hstate *h = hstate_vma(vma);
	/*
	 * Fix CVE-2019-11487
	 *  +1 line
	 */
	int err = -EFAULT;

	while (vaddr < vma->vm_end && remainder) {
		pte_t *pte;
		spinlock_t *ptl = NULL;
		int absent;
		struct page *page;

		/*
		 * Some archs (sparc64, sh*) have multiple pte_ts to
		 * each hugepage.  We have to make sure we get the
		 * first, for the page indexing below to work.
		 *
		 * Note that page table lock is not held when pte is null.
		 */
		pte = kgr_huge_pte_offset(mm, vaddr & huge_page_mask(h));
		if (pte)
			ptl = huge_pte_lock(h, mm, pte);
		absent = !pte || huge_pte_none(huge_ptep_get(pte));

		/*
		 * When coredumping, it suits get_dump_page if we just return
		 * an error where there's an empty slot with no huge pagecache
		 * to back it.  This way, we avoid allocating a hugepage, and
		 * the sparse dumpfile avoids allocating disk blocks, but its
		 * huge holes still show up with zeroes where they need to be.
		 */
		if (absent && (flags & FOLL_DUMP) &&
		    !kgr_hugetlbfs_pagecache_present(h, vma, vaddr)) {
			if (pte)
				spin_unlock(ptl);
			remainder = 0;
			break;
		}

		/*
		 * We need call hugetlb_fault for both hugepages under migration
		 * (in which case hugetlb_fault waits for the migration,) and
		 * hwpoisoned hugepages (in which case we need to prevent the
		 * caller from accessing to them.) In order to do this, we use
		 * here is_swap_pte instead of is_hugetlb_entry_migration and
		 * is_hugetlb_entry_hwpoisoned. This is because it simply covers
		 * both cases, and because we can't follow correct pages
		 * directly from any kind of swap entries.
		 */
		if (absent || is_swap_pte(huge_ptep_get(pte)) ||
		    ((flags & FOLL_WRITE) &&
		      !huge_pte_write(huge_ptep_get(pte)))) {
			int ret;

			if (pte)
				spin_unlock(ptl);
			ret = kgr_hugetlb_fault(mm, vma, vaddr,
				(flags & FOLL_WRITE) ? FAULT_FLAG_WRITE : 0);
			if (!(ret & VM_FAULT_ERROR))
				continue;

			remainder = 0;
			break;
		}

		pfn_offset = (vaddr & ~huge_page_mask(h)) >> PAGE_SHIFT;
		page = pte_page(huge_ptep_get(pte));

		/*
		 * Fix CVE-2019-11487
		 *  +13 lines
		 */
		/*
		 * Instead of doing 'try_get_page()' below in the same_page
		 * loop, just check the count once here.
		 */
		if (unlikely(page_count(page) <= 0)) {
			if (pages) {
				if (pte)
					spin_unlock(ptl);
				remainder = 0;
				err = -ENOMEM;
				break;
			}
		}
same_page:
		if (pages) {
			pages[i] = kgr_mem_map_offset(page, pfn_offset);
			/*
			 * Fix CVE-2019-11487
			 *  -1 line, +1 line
			 */
			kgr_get_page_foll(pages[i], false);
		}

		if (vmas)
			vmas[i] = vma;

		vaddr += PAGE_SIZE;
		++pfn_offset;
		--remainder;
		++i;
		if (vaddr < vma->vm_end && remainder &&
				pfn_offset < pages_per_huge_page(h)) {
			/*
			 * We use pfn_offset to avoid touching the pageframes
			 * of this compound page.
			 */
			goto same_page;
		}
		spin_unlock(ptl);
	}
	*nr_pages = remainder;
	*position = vaddr;

	/*
	 * Fix CVE-2019-11487
	 *  -1 line, +1 line
	 */
	return i ? i : err;
}
#else /* CONFIG_HUGETLBFS */

#define kgr_follow_hugetlb_page follow_hugetlb_page

#endif /* CONFIG_HUGETLBFS */

/* patched */
long kgr__get_user_pages(struct task_struct *tsk, struct mm_struct *mm,
			 unsigned long start, unsigned long nr_pages,
			 unsigned int gup_flags, struct page **pages,
			 struct vm_area_struct **vmas, int *nonblocking)
{
	long i;
	unsigned long vm_flags;
	unsigned int page_mask;

	if (!nr_pages)
		return 0;

	/* 
	 * Require read or write permissions.
	 * If FOLL_FORCE is set, we only require the "MAY" flags.
	 */
	vm_flags  = (gup_flags & FOLL_WRITE) ?
			(VM_WRITE | VM_MAYWRITE) : (VM_READ | VM_MAYREAD);
	vm_flags &= (gup_flags & FOLL_FORCE) ?
			(VM_MAYREAD | VM_MAYWRITE) : (VM_READ | VM_WRITE);

	VM_BUG_ON(!!pages != !!(gup_flags & FOLL_GET));

	/*
	 * If FOLL_FORCE is set then do not force a full fault as the hinting
	 * fault information is unrelated to the reference behaviour of a task
	 * using the address space
	 */
	if (!(gup_flags & FOLL_FORCE))
		gup_flags |= FOLL_NUMA;

	i = 0;

	do {
		struct vm_area_struct *vma;

		vma = kgr_find_extend_vma(mm, start);
		if (!vma && kgr_in_gate_area(mm, start)) {
			unsigned long pg = start & PAGE_MASK;
			pgd_t *pgd;
			pud_t *pud;
			pmd_t *pmd;
			pte_t *pte;

			/* user gate pages are read-only */
			if (gup_flags & FOLL_WRITE)
				return i ? : -EFAULT;
			if (pg > TASK_SIZE)
				pgd = kgr_pgd_offset_k(pg);
			else
				pgd = pgd_offset_gate(mm, pg);
			BUG_ON(pgd_none(*pgd));
			pud = pud_offset(pgd, pg);
			BUG_ON(pud_none(*pud));
			pmd = pmd_offset(pud, pg);
			if (pmd_none(*pmd))
				return i ? : -EFAULT;
			VM_BUG_ON(pmd_trans_huge(*pmd));
			pte = pte_offset_map(pmd, pg);
			if (pte_none(*pte)) {
				pte_unmap(pte);
				return i ? : -EFAULT;
			}
			vma = kgr_get_gate_vma(mm);
			if (pages) {
				struct page *page;

				page = kgr_vm_normal_page(vma, start, *pte);
				if (!page) {
					if (!(gup_flags & FOLL_DUMP) &&
					     is_zero_pfn(pte_pfn(*pte)))
						page = pte_page(*pte);
					else {
						pte_unmap(pte);
						return i ? : -EFAULT;
					}
				}
				/*
				 * Fix CVE-2019-11487
				 *  +4 lines
				 */
				if (!unlikely(!kgr_try_get_page(page))) {
					pte_unmap(pte);
					return i ? : -ENOMEM;
				}
				pages[i] = page;
				/*
				 * Fix CVE-2019-11487
				 *  -1 line
				 */
			}
			pte_unmap(pte);
			page_mask = 0;
			goto next_page;
		}

#ifdef CONFIG_XEN
		if (vma && (vma->vm_flags & VM_FOREIGN)) {
			struct vm_foreign_map *foreign_map =
				vma->vm_private_data;
			struct page **map = foreign_map->map;
			int offset = (start - vma->vm_start) >> PAGE_SHIFT;
			if (map[offset] != NULL) {
			        if (pages) {
			                struct page *page = map[offset];

					pages[i] = page;
					get_page(page);
				}
				if (vmas)
					vmas[i] = vma;
				i++;
				start += PAGE_SIZE;
				nr_pages--;
				continue;
			}
		}
#endif
		if (!vma ||
		    (vma->vm_flags & (VM_IO | VM_PFNMAP)) ||
		    !(vm_flags & vma->vm_flags))
			return i ? : -EFAULT;

		if (is_vm_hugetlb_page(vma)) {
			i = kgr_follow_hugetlb_page(mm, vma, pages, vmas,
					&start, &nr_pages, i, gup_flags);
			continue;
		}

		do {
			struct page *page;
			unsigned int foll_flags = gup_flags;
			unsigned int page_increm;

			/*
			 * If we have a pending SIGKILL, don't keep faulting
			 * pages and potentially allocating memory.
			 */
			if (unlikely(fatal_signal_pending(current)))
				return i ? i : -ERESTARTSYS;

			cond_resched();
			while (!(page = kgr_follow_page_mask(vma, start,
						foll_flags, &page_mask))) {
				int ret;
				unsigned int fault_flags = 0;

				/* mlock all present pages, but do not fault in new pages */
				if (foll_flags & FOLL_MLOCK)
					goto next_page;
				if (foll_flags & FOLL_WRITE)
					fault_flags |= FAULT_FLAG_WRITE;
				if (nonblocking)
					fault_flags |= FAULT_FLAG_ALLOW_RETRY;
				if (foll_flags & FOLL_NOWAIT)
					fault_flags |= (FAULT_FLAG_ALLOW_RETRY | FAULT_FLAG_RETRY_NOWAIT);

				ret = kgr_handle_mm_fault(mm, vma, start,
							  fault_flags);

				if (ret & VM_FAULT_ERROR) {
					if (ret & VM_FAULT_OOM)
						return i ? i : -ENOMEM;
					if (ret & (VM_FAULT_HWPOISON |
						   VM_FAULT_HWPOISON_LARGE)) {
						if (i)
							return i;
						else if (gup_flags & FOLL_HWPOISON)
							return -EHWPOISON;
						else
							return -EFAULT;
					}
					if (ret & (VM_FAULT_SIGBUS |
							VM_FAULT_SIGSEGV))
						return i ? i : -EFAULT;
					BUG();
				}

				if (tsk) {
					if (ret & VM_FAULT_MAJOR)
						tsk->maj_flt++;
					else
						tsk->min_flt++;
				}

				if (ret & VM_FAULT_RETRY) {
					if (nonblocking)
						*nonblocking = 0;
					return i;
				}

				/*
				 * The VM_FAULT_WRITE bit tells us that
				 * do_wp_page has broken COW when necessary,
				 * even if maybe_mkwrite decided not to set
				 * pte_write. We can thus safely do subsequent
				 * page lookups as if they were reads. But only
				 * do so when looping for pte_write is futile:
				 * in some cases userspace may also be wanting
				 * to write to the gotten user page, which a
				 * read fault here might prevent (a readonly
				 * page might get reCOWed by userspace write).
				 */
				if ((ret & VM_FAULT_WRITE) &&
				    !(vma->vm_flags & VM_WRITE))
					foll_flags |= FOLL_COW;

				cond_resched();
			}
			if (IS_ERR(page))
				return i ? i : PTR_ERR(page);
			if (pages) {
				pages[i] = page;

				flush_anon_page(vma, page, start);
				flush_dcache_page(page);
				page_mask = 0;
			}
next_page:
			if (vmas) {
				vmas[i] = vma;
				page_mask = 0;
			}
			page_increm = 1 + (~(start >> PAGE_SHIFT) & page_mask);
			if (page_increm > nr_pages)
				page_increm = nr_pages;
			i += page_increm;
			start += page_increm * PAGE_SIZE;
			nr_pages -= page_increm;
		} while (nr_pages && start < vma->vm_end);
	} while (nr_pages);
	return i;
}



int kgr_patch_bsc1133191_memory_init(void)
{
	return __kgr_resolve_kallsyms_relocs(kgr_funcs, ARRAY_SIZE(kgr_funcs));
}
