/*
 * kgraft_patch_bsc1133191_fuse
 *
 * Fix for CVE-2019-11487, bsc#1133191 (fuse.ko part)
 *
 *  Copyright (c) 2019 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pipe_fs_i.h>
#include <linux/splice.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <linux/slab.h>
#include "kgr_patch_bsc1133191_fuse.h"
#include "kgr_patch_bsc1133191_splice.h"
#include "kallsyms_relocs.h"

#if !IS_MODULE(CONFIG_FUSE_FS)
#error "Live patch supports only CONFIG_FUSE_FS=m"
#endif

#define KGR_PATCHED_MODULE "fuse"

struct fuse_conn;
struct fuse_copy_state;

static ssize_t (*kgr_fuse_dev_do_write)(struct fuse_conn *fud,
					struct fuse_copy_state *cs,
					size_t nbytes);

static struct kgr_kallsyms_reloc kgr_funcs[] = {
	{ "fuse_dev_do_write", (void *)&kgr_fuse_dev_do_write, "fuse" },
};


/* from fs/fuse/dev.c */
/* inlined */
static struct fuse_conn *kgr_fuse_get_conn(struct file *file)
{
	/*
	 * Lockless access is OK, because file->private data is set
	 * once during mount and is valid until the file is released.
	 */
	return file->private_data;
}

struct fuse_copy_state {
	struct fuse_conn *fc;
	int write;
	struct fuse_req *req;
	const struct iovec *iov;
	struct pipe_buffer *pipebufs;
	struct pipe_buffer *currbuf;
	struct pipe_inode_info *pipe;
	unsigned long nr_segs;
	unsigned long seglen;
	unsigned long addr;
	struct page *pg;
	void *mapaddr;
	void *buf;
	unsigned len;
	unsigned move_pages:1;
};

/* inlined */
static void kgr_fuse_copy_init(struct fuse_copy_state *cs, struct fuse_conn *fc,
			       int write,
			       const struct iovec *iov, unsigned long nr_segs)
{
	memset(cs, 0, sizeof(*cs));
	cs->fc = fc;
	cs->write = write;
	cs->iov = iov;
	cs->nr_segs = nr_segs;
}



/* patched */
ssize_t kgr_fuse_dev_splice_write(struct pipe_inode_info *pipe,
				  struct file *out, loff_t *ppos,
				  size_t len, unsigned int flags)
{
	unsigned nbuf;
	unsigned idx;
	struct pipe_buffer *bufs;
	struct fuse_copy_state cs;
	struct fuse_conn *fc;
	size_t rem;
	ssize_t ret;

	fc = kgr_fuse_get_conn(out);
	if (!fc)
		return -EPERM;

	bufs = kmalloc(pipe->buffers * sizeof(struct pipe_buffer), GFP_KERNEL);
	if (!bufs)
		return -ENOMEM;

	pipe_lock(pipe);
	nbuf = 0;
	rem = 0;
	for (idx = 0; idx < pipe->nrbufs && rem < len; idx++)
		rem += pipe->bufs[(pipe->curbuf + idx) & (pipe->buffers - 1)].len;

	ret = -EINVAL;
	if (rem < len) {
		/*
		 * Fix CVE-2019-11487
		 *  -2 lines, +1 line
		 */
		goto out_free;
	}

	rem = len;
	while (rem) {
		struct pipe_buffer *ibuf;
		struct pipe_buffer *obuf;

		BUG_ON(nbuf >= pipe->buffers);
		BUG_ON(!pipe->nrbufs);
		ibuf = &pipe->bufs[pipe->curbuf];
		obuf = &bufs[nbuf];

		if (rem >= ibuf->len) {
			*obuf = *ibuf;
			ibuf->ops = NULL;
			pipe->curbuf = (pipe->curbuf + 1) & (pipe->buffers - 1);
			pipe->nrbufs--;
		} else {
			/*
			 * Fix CVE-2019-11487
			 *  -1 line, +3 lines
			 */
			if (!kgr_pipe_buf_get(pipe, ibuf))
				goto out_free;

			*obuf = *ibuf;
			obuf->flags &= ~PIPE_BUF_FLAG_GIFT;
			obuf->len = rem;
			ibuf->offset += obuf->len;
			ibuf->len -= obuf->len;
		}
		nbuf++;
		rem -= obuf->len;
	}
	pipe_unlock(pipe);

	kgr_fuse_copy_init(&cs, fc, 0, NULL, nbuf);
	cs.pipebufs = bufs;
	cs.pipe = pipe;

	if (flags & SPLICE_F_MOVE)
		cs.move_pages = 1;

	ret = kgr_fuse_dev_do_write(fc, &cs, len);

	/*
	 * Fix CVE-2019-11487
	 *  +2 lines
	 */
	pipe_lock(pipe);
out_free:
	for (idx = 0; idx < nbuf; idx++) {
		struct pipe_buffer *buf = &bufs[idx];
		buf->ops->release(pipe, buf);
	}
	/*
	 * Fix CVE-2019-11487
	 *  +1 line
	 */
	pipe_unlock(pipe);
	/*
	 * Fix CVE-2019-11487
	 *  -1 line
	 */
	kfree(bufs);
	return ret;
}


static int kgr_patch_bsc1133191_fuse_module_notify(struct notifier_block *nb,
						   unsigned long action,
						   void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	ret = __kgr_resolve_kallsyms_relocs(kgr_funcs, ARRAY_SIZE(kgr_funcs));
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1133191_fuse_module_nb = {
	.notifier_call = kgr_patch_bsc1133191_fuse_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1133191_fuse_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(KGR_PATCHED_MODULE)) {
		ret = __kgr_resolve_kallsyms_relocs(kgr_funcs,
						    ARRAY_SIZE(kgr_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1133191_fuse_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1133191_fuse_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1133191_fuse_module_nb);
}
