#!/usr/bin/env rspec

# Copyright (c) [2013-2020] SUSE LLC
#
# All Rights Reserved.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of version 2 of the GNU General Public License as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, contact SUSE LLC.
#
# To contact SUSE LLC about this file by physical or electronic mail, you may
# find current contact information at www.suse.com.

require_relative "test_helper"

Yast.import "PortAliases"

describe Yast::PortAliases do
  # /usr/etc/servcies content example
  #
  # EtherNet-IP-1      2222/tcp     # EtherNet/IP I/O ;; IANA assigned this well-formed service name as a replacement for "EtherNet/IP-1".  [Brian_Batke_2]
  # EtherNet/IP-1      2222/tcp     # EtherNet/IP I/O  [Brian_Batke_2]
  # EtherNet-IP-1      2222/udp     # EtherNet/IP I/O ;; IANA assigned this well-formed service name as a replacement for "EtherNet/IP-1".  [Brian_Batke_2]
  # EtherNet/IP-1      2222/udp     # EtherNet/IP I/O  [Brian_Batke_2]
  # rfb                5900/tcp     vnc-server # Remote Framebuffer  [Tristan_Richardson] [RFC6143]
  # rfb                5900/udp     vnc-server # Remote Framebuffer  [Tristan_Richardson] [RFC6143]
  # blocks             10288/tcp    # Blocks  [Carl_Malamud]
  # blocks             10288/udp    # Blocks  [Carl_Malamud]
  # cosir              10321/tcp    # Computer Op System Information Report  [Kevin_C_Barber]
  # #                  10321/udp    Reserved
  # bngsync            10439/udp    # BalanceNG session table synchronization protocol  [Inlab_Software_GmbH] [Thomas_G._Obermair]
  # #                  10439/tcp    Reserved
  # #                  10500/tcp    Reserved
  # hip-nat-t          10500/udp    # HIP NAT-Traversal  [RFC5770] [Ari_Keranen]
  # MOS-lower          10540/tcp    # MOS Media Object Metadata Port  [Eric_Thorniley]
  # MOS-lower          10540/udp    # MOS Media Object Metadata Port  [Eric_Thorniley]
  # MOS-upper          10541/tcp    # MOS Running Order Port  [Eric_Thorniley]
  # MOS-upper          10541/udp    # MOS Running Order Port  [Eric_Thorniley]
  # EtherNet-IP-2      44818/tcp    # EtherNet/IP messaging ;; IANA assigned this well-formed service name as a replacement for "EtherNet/IP-2".  [Brian_Batke_2]
  # EtherNet/IP-2      44818/tcp    # EtherNet/IP messaging  [Brian_Batke_2]
  # EtherNet-IP-2      44818/udp    # EtherNet/IP messaging ;; IANA assigned this well-formed service name as a replacement for "EtherNet/IP-2".  [Brian_Batke_2]
  # EtherNet/IP-2      44818/udp    # EtherNet/IP messaging  [Brian_Batke_2]

  # The output generated by `getent services` for the file example above
  let(:output) do
    <<~OUTPUT
      EtherNet-IP-1      2222/tcp
      EtherNet/IP-1      2222/tcp
      EtherNet-IP-1      2222/udp
      EtherNet/IP-1      2222/udp
      rfb                5900/tcp     vnc-server
      rfb                5900/udp     vnc-server
      blocks             10288/tcp
      blocks             10288/udp
      cosir              10321/tcp
      bngsync            10439/udp
      hip-nat-t          10500/udp
      MOS-lower          10540/tcp
      MOS-lower          10540/udp
      MOS-upper          10541/tcp
      MOS-upper          10541/udp
      EtherNet-IP-2      44818/tcp
      EtherNet/IP-2      44818/tcp
      EtherNet-IP-2      44818/udp
      EtherNet/IP-2      44818/udp
    OUTPUT
  end

  let(:executor) { instance_double(Yast::Execute, on_target!: output) }

  before do
    allow(Yast::Execute).to receive(:stdout).and_return(executor)
  end

  describe ".IsAllowedPortName" do
    context "when nil is given" do
      it "logs an error" do
        expect(subject.log).to receive(:error).with(/Invalid/)

        subject.IsAllowedPortName(nil)
      end

      it "returns false" do
        expect(subject.IsAllowedPortName(nil)).to eq(false)
      end
    end

    context "when a number is given" do
      context "within the valid port range" do
        it "returns true" do
          expect(subject.IsAllowedPortName("65535")).to eq(true)
        end
      end

      context "beyond the upper limit" do
        it "returns false" do
          expect(subject.IsAllowedPortName("65536")).to eq(false)
        end
      end
      context "below the lower limit" do
        it "returns false" do
          expect(subject.IsAllowedPortName("-1")).to eq(false)
        end
      end
    end

    context "when a name is given" do
      context "containing only valid chars" do
        it "returns true" do
          expect(subject.IsAllowedPortName("valid-service.name-or-alias")).to eq(true)
        end
      end

      context "containing not valid chars" do
        it "returns false" do
          expect(subject.IsAllowedPortName("Not valid service alias")).to eq(false)
        end
      end
    end
  end

  describe ".AllowedPortNameOrNumber" do
    it "returns an informing message" do
      message = subject.AllowedPortNameOrNumber

      expect(message).to include("a-z")
      expect(message).to include("A-Z")
      expect(message).to include("0-9")
      expect(message).to include("*+._-")
      expect(message).to include("0 to 65535")
    end
  end

  describe ".GetListOfServiceAliases" do
    context "when a port number is given" do
      context "and there is a service for such port number" do
        let(:port_number) { "10541" }

        it "returns a list holding both, the port number and aliases" do
          expect(subject.GetListOfServiceAliases(port_number)).to eq(["10541", "MOS-upper"])
        end
      end

      context "but there is not a service for such port number" do
        let(:port_number) { "666" }

        it "returns a list holding only given argument" do
          expect(subject.GetListOfServiceAliases(port_number)).to eq([port_number])
        end
      end
    end

    context "when a port name is given" do
      context "and its an allowed port name" do
        context "and there is a service for such port name" do
          let(:port_name) { "MOS-upper" }

          it "returns a list holding both, given name and its port number" do
            expect(subject.GetListOfServiceAliases(port_name)).to eq(["10541", "MOS-upper"])
          end
        end

        context "but there is not a service for such port number" do
          let(:port_name) { "SomethingWrong" }

          it "returns a list holding only given argument" do
            expect(subject.GetListOfServiceAliases(port_name)).to eq([port_name])
          end
        end

        context "but it is not an allowed port name" do
          let(:port_name) { "Something Not Allowed" }

          it "logs an error" do
            expect(subject.log).to receive(:error).with(/not allowed/)

            subject.GetListOfServiceAliases(port_name)
          end

          it "returns a list holding only given argument" do
            expect(subject.GetListOfServiceAliases(port_name)).to eq([port_name])
          end
        end
      end
    end
  end

  describe ".IsKnownPortName" do
    context "when a known port name is given" do
      it "returns true" do
        expect(subject.IsKnownPortName("blocks")).to eq(true)
      end
    end

    context "when a known port alias is given" do
      it "returns true" do
        expect(subject.IsKnownPortName("vnc-server")).to eq(true)
      end
    end

    context "when an unknown port name is given" do
      it "returns false" do
        expect(subject.IsKnownPortName("unknown-name-or-alias")).to eq(false)
      end
    end
  end

  describe ".GetPortNumber" do
    context "when a port number is given" do
      it "returns the port number as an Integer" do
        expect(subject.GetPortNumber("80")).to eq(80)
      end
    end

    context "when a port name is given" do
      context "and the port is known" do
        it "returns its port number" do
          expect(subject.GetPortNumber("vnc-server")).to eq(5900)
        end
      end

      context "but the port is unknown" do
        it "returns nil" do
          expect(subject.GetPortNumber("Unknown-service")).to be_nil
        end
      end
    end
  end
end
