/*
 * Copyright 2020 Intel Corporation
 * SPDX-License-Identifier: Apache 2.0
 */

/*!
 * \file
 * \brief This file implements CSR generation for ECDSA
 */

#include "fdotypes.h"
#include "fdoCryptoHal.h"
#include "util.h"
#include "safe_lib.h"
#include "se_config.h"
#include <atca_basic.h>
#include <atcacert/atcacert_client.h>

#define SE_CSR_SIZE BUFF_SIZE_512_BYTES
#define SE_CSR_TEMPLATE_ID 0x3
#define SE_CSR_PK_HASH_ID SNSRC_PUB_KEY_HASH
#define SE_CSR_PK_BYTE_LOCATION 64
#define SE_CSR_LOCATION_SELECT DEVZONE_NONE


const uint8_t csr_template_device[] = {
    0x30, 0x82, 0x01, 0x1f, 0x30, 0x81, 0xc7, 0x02,
    0x01, 0x00, 0x30, 0x65, 0x31, 0x0b, 0x30, 0x09,
    0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49,
    0x4e, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
    0x04, 0x08, 0x0c, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
    0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x0c,
    0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
    0x03, 0x53, 0x44, 0x4f, 0x31, 0x13, 0x30, 0x11,
    0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0a, 0x43,
    0x6c, 0x69, 0x65, 0x6e, 0x74, 0x2d, 0x53, 0x44,
    0x4b, 0x31, 0x1e, 0x30, 0x1c, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x15, 0x44, 0x65, 0x76, 0x69,
    0x63, 0x65, 0x2d, 0x53, 0x44, 0x4f, 0x2d, 0x43,
    0x6c, 0x69, 0x65, 0x6e, 0x74, 0x2d, 0x53, 0x44,
    0x4b, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
    0x03, 0x42, 0x00, 0x04, 0xad, 0xef, 0x17, 0x96,
    0xf7, 0x3b, 0x08, 0x2f, 0xcf, 0xc4, 0x2d, 0x8f,
    0x75, 0x99, 0x9a, 0x29, 0xd8, 0x2f, 0x9c, 0x58,
    0x6e, 0xbf, 0xf8, 0xff, 0x92, 0x96, 0x99, 0x28,
    0x80, 0x8a, 0xe7, 0x4d, 0xb9, 0x1c, 0x7a, 0xe2,
    0x4b, 0x7c, 0xc7, 0xb3, 0x7a, 0x14, 0xfc, 0x83,
    0x52, 0x6e, 0xff, 0x51, 0xfc, 0xa9, 0x87, 0x05,
    0xe6, 0xd2, 0x14, 0xf7, 0x35, 0x48, 0xbf, 0x34,
    0xce, 0x83, 0x72, 0x7e, 0xa0, 0x00, 0x30, 0x0a,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04,
    0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02,
    0x20, 0x15, 0xba, 0x7f, 0xb8, 0x94, 0xae, 0x9c,
    0x47, 0x11, 0x6f, 0xa8, 0x7d, 0x58, 0xc5, 0x43,
    0x98, 0xd1, 0xe8, 0x19, 0x5c, 0x6e, 0x4f, 0xa7,
    0xec, 0x6d, 0xd2, 0x34, 0x51, 0xe3, 0x33, 0x3b,
    0x93, 0x02, 0x20, 0x13, 0xe9, 0x05, 0x51, 0x11,
    0x6f, 0x18, 0x5e, 0x91, 0x6f, 0x12, 0x78, 0x0d,
    0x4f, 0x52, 0x27, 0xa0, 0xd5, 0xd6, 0x6e, 0x30,
    0xb7, 0x7e, 0x88, 0xbc, 0x49, 0xf8, 0x50, 0x14,
    0x8e, 0x82, 0x64
};

const atcacert_def_t csr_def_device = {
    .type                   = 0, /* x509 certificate */
    .template_id            = SE_CSR_TEMPLATE_ID,
    .chain_id               = 0,
    .private_key_slot       = 0,
    .sn_source              = SE_CSR_PK_HASH_ID,
    .cert_sn_dev_loc        = {
		.zone      = SE_CSR_LOCATION_SELECT,
		.slot      = 0,
		.is_genkey = 0,
		.offset    = 0,
		.count     = 0
	},
    .issue_date_format      = 1,
    .expire_date_format     = 1,
    .tbs_cert_loc           = {
		.offset = 4,
		.count  = 202
	},
    .expire_years           = 0,
    .public_key_dev_loc     = {
		.zone      = SE_CSR_LOCATION_SELECT,
		.slot      = 0,
		.is_genkey = 1,
		.offset    = 0,
		.count     = SE_CSR_PK_BYTE_LOCATION
	},
    .comp_cert_dev_loc      = {
		.zone      = SE_CSR_LOCATION_SELECT,
		.slot      = 0,
		.is_genkey = 0,
		.offset    = 0,
		.count     = 0
    },
    .std_cert_elements      = {
		{ /* PUBLIC_KEY */
			.offset = 140,
			.count  = 64
		},
		{ /* SIGNATURE */
			.offset = 218,
			.count  = 73
		}
    },
    .cert_elements          = NULL,
    .cert_elements_count    = 0,
    .cert_template          = csr_template_device,
    .cert_template_size     = sizeof(csr_template_device)
};

/**
 * fdo_get_device_csr() - get the device CSR
 */
int32_t crypto_hal_get_device_csr(fdo_byte_array_t **csr)
{
	int ret = -1;

	char *csr_tmp = fdo_alloc(SE_CSR_SIZE);
	fdo_byte_array_t *csr_byte_arr = NULL;

	if (!csr_tmp) {
		LOG(LOG_ERROR, "Failed to allocate data.\n");
		goto err;
	}
        size_t csr_size = SE_CSR_SIZE;

	/* Creates a DER formatted CSR from the SE. */
	if (ATCA_SUCCESS !=
	    atcacert_create_csr_pem(&csr_def_device, csr_tmp, &csr_size)) {
		LOG(LOG_ERROR, " CSR generation failed\n");
		goto err;
	}

	/* Allocate byte array to send back data to DI state machine */
	csr_byte_arr = fdo_byte_array_alloc(csr_size);

	if (!csr_byte_arr) {
		LOG(LOG_ERROR,
		    "Failed to allocate data for storing csr data\n");
		goto err;
	}

	ret = memcpy_s(csr_byte_arr->bytes, csr_size, csr_tmp, csr_size);
	if (ret) {
		LOG(LOG_ERROR, "Failed to copy csr data\n");
		goto err;
	}
	ret = 0;
err:
	if ((NULL != csr_byte_arr) && (0 != ret)) {
		fdo_byte_array_free(csr_byte_arr);
		csr_byte_arr = NULL;
	}
	fdo_free(csr_tmp);

	*csr = csr_byte_arr;
	return ret;
}
