// Protocol Buffers - Google's data interchange format
// Copyright 2008 Google Inc.  All rights reserved.
// https://developers.google.com/protocol-buffers/
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// Author: kenton@google.com (Kenton Varda)
//  Based on original Protocol Buffers design by
//  Sanjay Ghemawat, Jeff Dean, and others.
//
// This file contains classes which describe a type of protocol message.
// You can use a message's descriptor to learn at runtime what fields
// it contains and what the types of those fields are.  The Message
// interface also allows you to dynamically access and modify individual
// fields by passing the FieldDescriptor of the field you are interested
// in.
//
// Most users will not care about descriptors, because they will write
// code specific to certain protocol types and will simply use the classes
// generated by the protocol compiler directly.  Advanced users who want
// to operate on arbitrary types (not known at compile time) may want to
// read descriptors in order to learn about the contents of a message.
// A very small number of users will want to construct their own
// Descriptors, either because they are implementing Message manually or
// because they are writing something like the protocol compiler.
//
// For an example of how you might use descriptors, see the code example
// at the top of message.h.

#ifndef GOOGLE_PROTOBUF_DESCRIPTOR_LEGACY_H__
#define GOOGLE_PROTOBUF_DESCRIPTOR_LEGACY_H__

#include "absl/strings/string_view.h"
#include "google/protobuf/descriptor.h"

// Must be included last.
#include "google/protobuf/port_def.inc"

// This file is meant to be a temporary housing for legacy descriptor APIs we
// want to deprecate and remove.  This will help prevent backslide by allowing
// us to control visibility.

namespace google {
namespace protobuf {
PROTOBUF_IGNORE_DEPRECATION_START

// Wraps FileDescriptor.
class PROTOBUF_EXPORT FileDescriptorLegacy {
 public:
  explicit FileDescriptorLegacy(const FileDescriptor* desc) : desc_(desc) {}

  // Any dependencies on file-level syntax keyword should be replaced by
  // feature-level switches to support go/protobuf-editions.
  enum Syntax {
    SYNTAX_UNKNOWN = FileDescriptor::SYNTAX_UNKNOWN,
    SYNTAX_PROTO2 = FileDescriptor::SYNTAX_PROTO2,
    SYNTAX_PROTO3 = FileDescriptor::SYNTAX_PROTO3,
  };
  Syntax syntax() const { return static_cast<Syntax>(desc_->syntax()); }
  static absl::string_view SyntaxName(Syntax syntax) {
    return FileDescriptor::SyntaxName(
        static_cast<FileDescriptor::Syntax>(syntax));
  }

 private:
  const FileDescriptor* desc_;
};

class PROTOBUF_EXPORT FieldDescriptorLegacy {
 public:
  explicit FieldDescriptorLegacy(const FieldDescriptor* desc) : desc_(desc) {}

  bool has_optional_keyword() const { return desc_->has_optional_keyword(); }

 private:
  const FieldDescriptor* desc_;
};

class PROTOBUF_EXPORT OneofDescriptorLegacy {
 public:
  explicit OneofDescriptorLegacy(const OneofDescriptor* desc) : desc_(desc) {}

  bool is_synthetic() const { return desc_->is_synthetic(); }

 private:
  const OneofDescriptor* desc_;
};

PROTOBUF_IGNORE_DEPRECATION_STOP
}  // namespace protobuf
}  // namespace google

#include "google/protobuf/port_undef.inc"

#endif  // GOOGLE_PROTOBUF_DESCRIPTOR_LEGACY_H__
