/*
 * livepatch_bsc1232927
 *
 * Fix for CVE-2024-50124, bsc#1232927
 *
 *  Upstream commit:
 *  246b435ad668 ("Bluetooth: ISO: Fix UAF on iso_sock_timeout")
 *
 *  SLE12-SP5 commit:
 *  Not affected
 *
 *  SLE15-SP3 and -SP4 commit:
 *  Not affected
 *
 *  SLE15-SP5 commit:
 *  a1432cecd30940becf85d417bf9f362db2680712
 *
 *  SLE15-SP6 commit:
 *  25f5727fe1a3d7ae5c0c6fc5a76265de0a78bb45
 *
 *  SLE MICRO-6-0 commit:
 *  25f5727fe1a3d7ae5c0c6fc5a76265de0a78bb45
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_BT)

#if !IS_MODULE(CONFIG_BT)
#error "Live patch supports only CONFIG=m"
#endif

/* klp-ccp: from net/bluetooth/iso.c */
#include <linux/module.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>

#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>

/* klp-ccp: from include/net/bluetooth/iso.h */
#define ISO_MAX_NUM_BIS		0x1f

/* klp-ccp: from net/bluetooth/iso.c */
struct iso_conn {
	struct hci_conn	*hcon;

	/* @lock: spinlock protecting changes to iso_conn fields */
	spinlock_t	lock;
	struct sock	*sk;

	struct delayed_work	timeout_work;

	struct sk_buff	*rx_skb;
	__u32		rx_len;
	__u16		tx_sn;
};

#define iso_conn_lock(c)	spin_lock(&(c)->lock)
#define iso_conn_unlock(c)	spin_unlock(&(c)->lock)

#define iso_pi(sk) ((struct iso_pinfo *)sk)

#define EIR_SERVICE_DATA_LENGTH 4
#define BASE_MAX_LENGTH (HCI_MAX_PER_AD_LENGTH - EIR_SERVICE_DATA_LENGTH)

enum {
	BT_SK_BIG_SYNC,
	BT_SK_PA_SYNC,
	BT_SK_PA_SYNC_TERM,
};

struct iso_pinfo {
	struct bt_sock		bt;
	bdaddr_t		src;
	__u8			src_type;
	bdaddr_t		dst;
	__u8			dst_type;
	__u8			bc_sid;
	__u8			bc_num_bis;
	__u8			bc_bis[ISO_MAX_NUM_BIS];
	__u16			sync_handle;
	unsigned long		flags;
	struct bt_iso_qos	qos;
	bool			qos_user_set;
	__u8			base_len;
	__u8			base[BASE_MAX_LENGTH];
	struct iso_conn		*conn;
};

typedef bool (*iso_sock_match_t)(struct sock *sk, void *data);

extern struct sock *iso_get_sock_listen(bdaddr_t *src, bdaddr_t *dst,
					iso_sock_match_t match, void *data);

extern struct bt_sock_list iso_sk_list;

#include "../bsc1232929/livepatch_bsc1232929.h"

static struct sock *iso_sock_hold(struct iso_conn *conn)
{
	if (!conn || !bt_sock_linked(&iso_sk_list, conn->sk))
		return NULL;

	sock_hold(conn->sk);

	return conn->sk;
}

void klpp_iso_sock_timeout(struct work_struct *work)
{
	struct iso_conn *conn = container_of(work, struct iso_conn,
					     timeout_work.work);
	struct sock *sk;

	iso_conn_lock(conn);
	sk = iso_sock_hold(conn);
	iso_conn_unlock(conn);

	if (!sk)
		return;

	BT_DBG("sock %p state %d", sk, sk->sk_state);

	lock_sock(sk);
	sk->sk_err = ETIMEDOUT;
	sk->sk_state_change(sk);
	release_sock(sk);
	sock_put(sk);
}

extern void iso_sock_clear_timer(struct sock *sk);

extern void iso_chan_del(struct sock *sk, int err);

extern bool iso_match_conn_sync_handle(struct sock *sk, void *data);

void klpp_iso_conn_del(struct hci_conn *hcon, int err)
{
	struct iso_conn *conn = hcon->iso_data;
	struct sock *sk;
	struct sock *parent;

	if (!conn)
		return;

	BT_DBG("hcon %p conn %p, err %d", hcon, conn, err);

	/* Kill socket */
	iso_conn_lock(conn);
	sk = iso_sock_hold(conn);
	iso_conn_unlock(conn);

	if (sk) {
		lock_sock(sk);

		/* While a PA sync hcon is in the process of closing,
		 * mark parent socket with a flag, so that any residual
		 * BIGInfo adv reports that arrive before PA sync is
		 * terminated are not processed anymore.
		 */
		if (test_bit(BT_SK_PA_SYNC, &iso_pi(sk)->flags)) {
			parent = iso_get_sock_listen(&hcon->src,
						     &hcon->dst,
						     iso_match_conn_sync_handle,
						     hcon);

			if (parent) {
				set_bit(BT_SK_PA_SYNC_TERM,
					&iso_pi(parent)->flags);
				sock_put(parent);
			}
		}

		iso_sock_clear_timer(sk);
		iso_chan_del(sk, err);
		release_sock(sk);
		sock_put(sk);
	}

	/* Ensure no more work items will run before freeing conn. */
	cancel_delayed_work_sync(&conn->timeout_work);

	hcon->iso_data = NULL;
	kfree(conn);
}

#include "livepatch_bsc1232927.h"

#include <linux/livepatch.h>

extern typeof(iso_chan_del) iso_chan_del
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, iso_chan_del);
extern typeof(iso_get_sock_listen) iso_get_sock_listen
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, iso_get_sock_listen);
extern typeof(iso_match_conn_sync_handle) iso_match_conn_sync_handle
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, iso_match_conn_sync_handle);
extern typeof(iso_sk_list) iso_sk_list
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, iso_sk_list);
extern typeof(iso_sock_clear_timer) iso_sock_clear_timer
	 KLP_RELOC_SYMBOL(bluetooth, bluetooth, iso_sock_clear_timer);

#endif /* IS_ENABLED(CONFIG_BT) */
