#!/bin/bash
# Script for generating Fedora releng release signing tickets.  Generated by
# https://github.com/coreos/repo-templates; do not edit downstream.

set -euo pipefail

usage() {
    echo "Usage: $0 test <version-release> <dir>"
    echo "Usage: $0 ticket <version-release>"
    exit 1
}

make_script() {
    sed -e "s/@@VERSION@@/$ver/g" -e "s/@@RELEASE@@/$rel/g" <<'EOF'
#!/bin/bash
set -eux -o pipefail

# Use the Fedora 38 key for the detached signatures
KEYTOSIGNWITH='fedora-38'

VR='@@VERSION@@-@@RELEASE@@.fc38'
RPMKEY='eb10b464' # Fedora 38 key

do_sign() {
    # Sign with sigul unless FAKESIGN=1
    if [ ${FAKESIGN:-0} != 1 ]; then
        sigul sign-data -a $KEYTOSIGNWITH "$1" -o "$1.asc"
    else
        echo INVALID > "$1.asc"
    fi
}

# Grab the binaries out of the redistributable rpm
rpm="ignition-validate-redistributable-${VR}.noarch.rpm"
koji download-build --key $RPMKEY --rpm $rpm
rpm -qip $rpm | grep -P "^Signature.*${RPMKEY}$" # Verify the output has the key in it
rpm2cpio $rpm | cpio -idv './usr/share/ignition/ignition-validate-*'

# Rename the binaries
mv usr/share/ignition/ignition-validate-aarch64-apple-darwin \
    ignition-validate-aarch64-apple-darwin
mv usr/share/ignition/ignition-validate-aarch64-unknown-linux-gnu-static \
    ignition-validate-aarch64-linux
mv usr/share/ignition/ignition-validate-ppc64le-unknown-linux-gnu-static \
    ignition-validate-ppc64le-linux
mv usr/share/ignition/ignition-validate-s390x-unknown-linux-gnu-static \
    ignition-validate-s390x-linux
mv usr/share/ignition/ignition-validate-x86_64-apple-darwin \
    ignition-validate-x86_64-apple-darwin
mv usr/share/ignition/ignition-validate-x86_64-pc-windows-gnu.exe \
    ignition-validate-x86_64-pc-windows-gnu.exe
mv usr/share/ignition/ignition-validate-x86_64-unknown-linux-gnu-static \
    ignition-validate-x86_64-linux

# Sign them
do_sign ignition-validate-aarch64-apple-darwin
do_sign ignition-validate-aarch64-linux
do_sign ignition-validate-ppc64le-linux
do_sign ignition-validate-s390x-linux
do_sign ignition-validate-x86_64-apple-darwin
do_sign ignition-validate-x86_64-pc-windows-gnu.exe
do_sign ignition-validate-x86_64-linux

# Fix permissions and clean up
chmod go+r *.asc
rm $rpm; rmdir ./usr/share/ignition; rmdir ./usr/share; rmdir ./usr
EOF
}

make_ticket() {
    sed "s/@@VERSION@@/$ver/g" <<'EOF'
TITLE: Create detached signatures for the ignition @@VERSION@@ release

Please create detached signatures for the binaries we will upload to GitHub for the `ignition` @@VERSION@@ release.  This is a manual process for now, pending the automation discussed in https://pagure.io/robosignatory/issue/53 and https://github.com/coreos/fedora-coreos-tracker/issues/335.

The binaries themselves have been built in koji.  Here is a small script to grab all of the rpms and the files out of the rpms and name them appropriately:

```
EOF
    make_script
    cat <<'EOF'
```

After running this you should end up with a directory with files in it like:

```
$ ls -1
ignition-validate-aarch64-apple-darwin
ignition-validate-aarch64-apple-darwin.asc
ignition-validate-aarch64-linux
ignition-validate-aarch64-linux.asc
ignition-validate-ppc64le-linux
ignition-validate-ppc64le-linux.asc
ignition-validate-s390x-linux
ignition-validate-s390x-linux.asc
ignition-validate-x86_64-apple-darwin
ignition-validate-x86_64-apple-darwin.asc
ignition-validate-x86_64-pc-windows-gnu.exe
ignition-validate-x86_64-pc-windows-gnu.exe.asc
ignition-validate-x86_64-linux
ignition-validate-x86_64-linux.asc
```
EOF
}

[ "$#" -lt 2 ] && usage
cmd="$1"
ver="$2"
# Disallow version with preceding "v"
echo "$ver" | grep -q "v" && usage
# Require version-release
echo "$ver" | grep -q "-" || usage
rel="${ver#*-}"
ver="${ver%%-*}"

case "$cmd" in
test)
    [ "$#" != 3 ] && usage
    dir="$3"
    mkdir "$dir"
    make_script > "$dir/script"
    cd "$dir" && FAKESIGN=1 bash script
    ;;
ticket)
    make_ticket
    ;;
*)
    usage
    ;;
esac
