// SPDX-FileCopyrightText: 2025 SUSE LLC
//
// SPDX-License-Identifier: Apache-2.0

package templates

import (
	"io"
	"text/template"

	"github.com/uyuni-project/uyuni-tools/shared/types"
)

const httpdTemplate = `# uyuni-proxy-httpd.service, generated by mgrpxy
# Use an uyuni-proxy-httpd.service.d/local.conf file to override

[Unit]
Description=Uyuni proxy httpd container service
Wants=network.target
After=network-online.target
BindsTo=uyuni-proxy-pod.service
After=uyuni-proxy-pod.service

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
{{- if .HTTPProxyFile }}
EnvironmentFile={{ .HTTPProxyFile }}
{{- end }}
Restart=on-failure
ExecStartPre=/bin/rm -f %t/uyuni-proxy-httpd.pid %t/uyuni-proxy-httpd.ctr-id

ExecStart=/bin/sh -c '/usr/bin/podman run \
	--conmon-pidfile %t/uyuni-proxy-httpd.pid \
	--cidfile %t/uyuni-proxy-httpd.ctr-id \
	--cgroups=no-conmon \
	--pod-id-file %t/uyuni-proxy-pod.pod-id -d \
	--replace -dt \
	-v /etc/uyuni/proxy:/etc/uyuni:ro \
	{{- range .Volumes }}
	-v {{ .Name }}:{{ .MountPath }} \
	{{- end }}
	{{- if .SystemIDSecret }}
	--secret {{ .SystemIDSecret }},type=mount,mode=0444,target="/etc/sysconfig/rhn/systemid" \
	{{- end }}
	${HTTPD_EXTRA_CONF} --name uyuni-proxy-httpd \
	${UYUNI_IMAGE}'

ExecStop=/usr/bin/podman stop --ignore --cidfile %t/uyuni-proxy-httpd.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/uyuni-proxy-httpd.ctr-id
PIDFile=%t/uyuni-proxy-httpd.pid
TimeoutStopSec=60
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

// HttpdTemplateData represents HTTPD information to create systemd file.
type HttpdTemplateData struct {
	Volumes        []types.VolumeMount
	HTTPProxyFile  string
	SystemIDSecret string
}

// Render will create the systemd configuration file.
func (data HttpdTemplateData) Render(wr io.Writer) error {
	t := template.Must(template.New("service").Parse(httpdTemplate))
	return t.Execute(wr, data)
}
