// SPDX-FileCopyrightText: 2024 SUSE LLC
//
// SPDX-License-Identifier: Apache-2.0

package kubernetes

import (
	"github.com/spf13/cobra"
	"github.com/uyuni-project/uyuni-tools/mgrpxy/shared/kubernetes"
	pxy_utils "github.com/uyuni-project/uyuni-tools/mgrpxy/shared/utils"
	. "github.com/uyuni-project/uyuni-tools/shared/l10n"
	"github.com/uyuni-project/uyuni-tools/shared/types"
	"github.com/uyuni-project/uyuni-tools/shared/utils"
)

type kubernetesProxyInstallFlags struct {
	pxy_utils.ProxyImageFlags `mapstructure:",squash"`
	Helm                      kubernetes.HelmFlags
	SCC                       types.SCCCredentials
}

func newCmd(globalFlags *types.GlobalFlags, run utils.CommandFunc[kubernetesProxyInstallFlags]) *cobra.Command {
	cmd := &cobra.Command{
		Use:   "kubernetes [path/to/config.tar.gz]",
		Short: L("Install a new proxy on a running kubernetes cluster"),
		Long: L(`Install a new proxy on a running kubernetes cluster.

It only takes the path to the configuration tarball generated by the server
as parameter.

The install kubernetes command assumes kubectl is installed locally.

NOTE: for now installing on a remote kubernetes cluster is not supported!
`),
		Args: cobra.ExactArgs(1),
		RunE: func(cmd *cobra.Command, args []string) error {
			var flags kubernetesProxyInstallFlags
			return utils.CommandHelper(globalFlags, cmd, args, &flags, nil, run)
		},
	}

	pxy_utils.AddImageFlags(cmd)
	pxy_utils.AddSCCFlag(cmd)
	kubernetes.AddHelmFlags(cmd)

	return cmd
}

// NewCommand install a new proxy on a running kubernetes cluster.
func NewCommand(globalFlags *types.GlobalFlags) *cobra.Command {
	return newCmd(globalFlags, installForKubernetes)
}
