// SPDX-FileCopyrightText: 2025 SUSE LLC
//
// SPDX-License-Identifier: Apache-2.0

package templates

import (
	"io"
	"text/template"

	"github.com/uyuni-project/uyuni-tools/shared/types"
)

const salineServiceTemplate = `
# uyuni-saline.service, generated by mgradm
# Use an uyuni-saline.service.d/custom.conf file to override
[Unit]
Description=Uyuni Saline image container service
Wants=network.target uyuni-server.service
Requires=uyuni-server.service
After=network-online.target uyuni-server.service
RequiresMountsFor=%t/containers

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=on-failure
ExecStartPre=/bin/rm -f %t/uyuni-saline-%i.pid %t/%n.ctr-id
ExecStartPre=/usr/bin/podman rm --ignore --force -t 10 {{ .NamePrefix }}-saline-%i
ExecStart=/bin/sh -c '/usr/bin/podman run \
	--conmon-pidfile %t/uyuni-saline-%i.pid \
	--cidfile=%t/%n-%i.ctr-id \
	--cgroups=no-conmon \
	--sdnotify=conmon \
	--security-opt label=type:container_init_t \
	-d \
	--name {{ .NamePrefix }}-saline-%i \
	--hostname {{ .NamePrefix }}-saline-%i.mgr.internal \
	--network-alias saline \
	--network {{ .Network }} \
	{{- range .Volumes }}
	-v {{ .Name }}:{{ .MountPath }} \
	{{- end }}
	-e TZ=${TZ} \
	-e NOSSL=YES \
	${UYUNI_SALINE_IMAGE}'
ExecStop=/usr/bin/podman stop --ignore -t 10 --cidfile=%t/%n-%i.ctr-id
ExecStopPost=/usr/bin/podman rm -f --ignore -t 10 --cidfile=%t/%n-%i.ctr-id
PIDFile=%t/uyuni-saline-%i.pid
TimeoutStopSec=20
TimeoutStartSec=10
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

// SalineServiceTemplateData holds information to create systemd file for saline container.
type SalineServiceTemplateData struct {
	NamePrefix string
	Image      string
	Network    string
	Volumes    []types.VolumeMount
}

// Render will create the systemd configuration file.
func (data SalineServiceTemplateData) Render(wr io.Writer) error {
	t := template.Must(template.New("service").Parse(salineServiceTemplate))
	return t.Execute(wr, data)
}
