// SPDX-FileCopyrightText: 2025 SUSE LLC
//
// SPDX-License-Identifier: Apache-2.0

package templates

import (
	"io"
	"text/template"

	"github.com/uyuni-project/uyuni-tools/shared/types"
)

const pgsqlServiceTemplate = `# uyuni-db-server.service, generated by mgradm
# Use an uyuni-db-server.service.d/local.conf file to override

[Unit]
Description=Uyuni database container service
Wants=network.target
After=network-online.target
RequiresMountsFor=%t/containers

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=on-failure
ExecStartPre=/bin/rm -f %t/%n.pid %t/%n.ctr-id
ExecStartPre=/usr/bin/podman rm --ignore --force -t 10 {{ .NamePrefix }}-db
ExecStart=/bin/sh -c '/usr/bin/podman run \
	--conmon-pidfile %t/%n.pid \
	--cidfile=%t/%n.ctr-id \
	--cgroups=no-conmon \
	--shm-size=0 \
	--shm-size-systemd=0 \
	--sdnotify=conmon \
	-d \
	--name {{ .NamePrefix }}-db \
	--hostname {{ .NamePrefix }}-db.mgr.internal \
	--network-alias db \
	--network-alias reportdb \
	--secret {{ .CaSecret }},type=mount,target={{ .CaPath }} \
	--secret {{ .KeySecret }},type=mount,uid=999,mode=0400,target={{ .KeyPath }} \
	--secret {{ .CertSecret }},type=mount,target={{ .CertPath }} \
	--secret {{ .AdminUser }},type=env,target=POSTGRES_USER \
	--secret {{ .AdminPassword }},type=env,target=POSTGRES_PASSWORD \
	--secret {{ .ManagerUser }},type=env,target=MANAGER_USER \
	--secret {{ .ManagerPassword }},type=env,target=MANAGER_PASS \
	--secret {{ .ReportUser }},type=env,target=REPORT_DB_USER \
	--secret {{ .ReportPassword }},type=env,target=REPORT_DB_PASS \
	{{- range .Ports }}
	-p {{ .Exposed }}:{{ .Port }}{{if .Protocol}}/{{ .Protocol }}{{end}} \
        {{- if $.IPV6Enabled }}
	-p [::]:{{ .Exposed }}:{{ .Port }}{{if .Protocol}}/{{ .Protocol }}{{end}} \
        {{- end }}
	{{- end }}
        {{- range .Volumes }}
        -v {{ .Name }}:{{ .MountPath }} \
        {{- end }}
	--network {{ .Network }} \
	${PODMAN_EXTRA_ARGS} ${UYUNI_IMAGE}'
ExecStop=/usr/bin/podman stop \
	--ignore -t 10 \
	--cidfile=%t/%n.ctr-id
ExecStopPost=/usr/bin/podman rm \
	-f \
	--ignore -t 10 \
	--cidfile=%t/%n.ctr-id

PIDFile=%t/%n.pid
TimeoutStopSec=180
TimeoutStartSec=900
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

// PostgresServiceTemplateData POD information to create systemd file.
type PgsqlServiceTemplateData struct {
	Volumes         []types.VolumeMount
	NamePrefix      string
	Ports           []types.PortMap
	Image           string
	Network         string
	IPV6Enabled     bool
	CaSecret        string
	CaPath          string
	CertSecret      string
	CertPath        string
	KeySecret       string
	KeyPath         string
	AdminUser       string
	AdminPassword   string
	ManagerUser     string
	ManagerPassword string
	ReportUser      string
	ReportPassword  string
}

// Render will create the systemd configuration file.
func (data PgsqlServiceTemplateData) Render(wr io.Writer) error {
	t := template.Must(template.New("service").Parse(pgsqlServiceTemplate))
	return t.Execute(wr, data)
}
