// SPDX-FileCopyrightText: 2025 SUSE LLC
//
// SPDX-License-Identifier: Apache-2.0

package templates

import (
	"fmt"
	"strings"
	"text/template"

	"github.com/uyuni-project/uyuni-tools/shared/kubernetes"
	. "github.com/uyuni-project/uyuni-tools/shared/l10n"
)

const certificate = `apiVersion: cert-manager.io/v1
kind: Certificate
metadata:
  name: {{ .SecretName }}
  namespace: {{ .Namespace }}
  labels:
    app: ` + kubernetes.ServerApp + `
spec:
  secretName: {{ .SecretName }}
  secretTemplate:
    labels:
      app: ` + kubernetes.ServerApp + `
  isCA: false
  usages:
  - server auth
  dnsNames:
  {{- range .DNSNames }}
    - {{ . }}
  {{- end }}
  issuerRef:
    name: ` + kubernetes.CAIssuerName + `
    kind: Issuer
    group: cert-manager.io
---
`

// CertificateData is the template rendering a cert-manager Certificate object using the uyuni CA Issuer.
type CertificateData struct {
	// Namespace is the Kubernetes namespace where to create the certificate object.
	Namespace string
	// SecretName is the name of the secret generated by the certificate object and will be used as its name too.
	SecretName string
	// DNSNames is a slice of DNS names to generate the SSL certificate for.
	// At least one has to be provided.
	DNSNames []string
}

// Render generates a string from the certificate data.
func (c CertificateData) Render() (out string, err error) {
	if len(c.DNSNames) == 0 {
		return "", fmt.Errorf(L("%s certificate needs at least one DNS entry"), c.SecretName)
	}

	builder := new(strings.Builder)
	t := template.Must(template.New("certificate").Parse(certificate))
	err = t.Execute(builder, c)
	if err == nil {
		out = builder.String()
	}

	return
}
