/*
 * Copyright (C) 2021 Andreas Steffen
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "test_suite.h"

#include <plugins/plugin_feature.h>

typedef struct rsa_oaep_test_t rsa_oaep_test_t;

struct rsa_oaep_test_t {
	chunk_t msg;
	chunk_t label;
	chunk_t ct;
};

/**
 * Private RSA key to load
 */
static chunk_t rsa_2048_key = chunk_from_chars( /* PKCS#8 format */
		0x30,0x82,0x04,0xbd,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x04,0xa7,0x30,0x82,0x04,0xa3,0x02,0x01,
		0x00,0x02,0x82,0x01,0x01,0x00,0xcd,0xa6,0xfa,0x5c,0xa7,0x6b,0xfe,0x04,0x92,0xec,
		0x57,0xe0,0xa3,0xbf,0xff,0x72,0x72,0xdc,0x8d,0x1e,0x25,0xad,0x1f,0xb3,0x38,0xaa,
		0x05,0x0f,0x02,0xc1,0x04,0xe6,0x31,0x33,0xd6,0xb5,0xb7,0xc4,0x98,0x5e,0xbb,0xae,
		0x9a,0xc0,0x36,0xa5,0xb9,0xc0,0x30,0x74,0xd6,0x0a,0xec,0x8e,0x25,0xba,0xf3,0x92,
		0xa0,0xc4,0x30,0xff,0x05,0xb8,0x8e,0x94,0x88,0x05,0xd3,0xdd,0x74,0x51,0x1d,0x88,
		0x85,0x25,0x0a,0x7b,0x57,0x42,0x15,0xad,0xa0,0x15,0xc5,0x59,0x07,0x66,0x86,0xe2,
		0x53,0xcc,0xc9,0x6c,0x08,0x15,0xb1,0x29,0x1e,0xe7,0x87,0xcc,0x33,0x63,0xb4,0xf7,
		0x7d,0x93,0x0e,0xb9,0x98,0xd7,0xc5,0x82,0xb2,0x4c,0xea,0x9c,0xe2,0x1d,0xe9,0x72,
		0x27,0x91,0x98,0x98,0x63,0xa2,0x7e,0xbc,0x80,0xa0,0x0d,0xe5,0xbd,0x2f,0x92,0x28,
		0x77,0x5e,0x5a,0x4c,0xeb,0x05,0x4d,0x58,0xc9,0xbe,0x36,0xa0,0x54,0x33,0x69,0x71,
		0xa1,0x36,0x42,0xdd,0x95,0x10,0xdd,0x69,0x6a,0xa2,0x68,0xdb,0x3a,0xab,0x22,0x99,
		0xd5,0xd8,0x8f,0x8e,0x56,0x24,0x34,0xd1,0x42,0x70,0x94,0xd3,0xdf,0x8e,0x72,0xd1,
		0xef,0x69,0xb4,0xed,0x34,0xd1,0x2b,0xac,0x37,0x52,0x23,0xb2,0xa2,0x5c,0xf2,0x27,
		0xf7,0x35,0xf8,0x16,0xe8,0x5e,0x17,0x23,0x93,0x04,0x76,0x9a,0x60,0x82,0x15,0x4c,
		0xd1,0x58,0x99,0xfc,0x1e,0xae,0xfb,0x69,0xb7,0x48,0xa3,0xe5,0xed,0x24,0xd3,0x83,
		0x72,0x59,0x7d,0xe3,0xe4,0xe2,0xa2,0x7b,0x95,0x1d,0x6a,0xc7,0xdb,0x18,0x2d,0x68,
		0x09,0xd8,0xff,0x51,0x1b,0x7f,0x02,0x03,0x01,0x00,0x01,0x02,0x82,0x01,0x00,0x5a,
		0x3d,0xc9,0x62,0xda,0xfc,0xa2,0x6c,0xb3,0x64,0x0e,0x73,0xbe,0xa7,0x43,0x9a,0x9f,
		0x18,0x74,0xbc,0x23,0xe0,0x42,0x26,0xcc,0xd8,0x9e,0x7b,0xa5,0xc3,0xb9,0x38,0xa1,
		0xa2,0x93,0xb7,0x0d,0xba,0xb0,0xf9,0xf0,0xf5,0x7f,0x66,0x95,0x14,0x47,0xdc,0x33,
		0xe7,0x30,0xfd,0x7e,0x2c,0x2a,0x16,0x4d,0x47,0xac,0x50,0x2b,0x07,0xdd,0x24,0xcd,
		0x3c,0x14,0x2c,0x2a,0x79,0xe4,0xce,0xab,0x5c,0xfa,0xbf,0xf4,0x47,0x87,0x54,0xb2,
		0x5a,0x8c,0x02,0xc1,0xa4,0x7d,0x80,0xd9,0xf3,0x7a,0xbe,0x44,0x2c,0xa9,0xa7,0x8b,
		0x23,0xf6,0x31,0xb6,0xff,0x3e,0x15,0xa4,0x95,0x6d,0x7f,0x18,0x59,0x0c,0xde,0xb2,
		0x06,0xd5,0xe2,0xb6,0x98,0xbd,0x08,0x4f,0x26,0x0e,0x82,0xef,0x28,0xff,0x9e,0xc6,
		0xdb,0xc8,0x5a,0x89,0x5e,0xc8,0xa3,0x86,0x57,0x50,0xf5,0x01,0xb9,0x61,0x25,0xdb,
		0x1b,0xbd,0xd9,0x9a,0x4a,0xe4,0x68,0x8a,0xdb,0x30,0x4a,0xab,0xdf,0xc4,0xe0,0xcf,
		0xb9,0xfe,0x6b,0x6b,0xc0,0xdb,0x74,0xc8,0x8a,0xf8,0x21,0x7e,0xed,0x73,0x8a,0x0d,
		0x04,0xfe,0x8d,0x32,0xc1,0xd1,0x10,0x37,0x0c,0xe1,0xc1,0xb2,0xf6,0x30,0x65,0x73,
		0x50,0x69,0x49,0x42,0x73,0x08,0x78,0xe6,0xff,0xf7,0x7a,0xda,0x7e,0x9a,0x31,0x7d,
		0xf8,0xbc,0x05,0x9e,0xa7,0x08,0x13,0x25,0x30,0x6b,0x8e,0xb2,0xfa,0x0d,0x3a,0x3d,
		0x89,0xfa,0xe4,0x76,0xd9,0x34,0x48,0x92,0xbc,0xd5,0xa4,0x2c,0xf8,0x3b,0x7b,0xcf,
		0x3e,0x0e,0x51,0xb4,0xc7,0x8e,0x72,0xb3,0x20,0x7a,0x60,0xa7,0x01,0xad,0xb1,0x02,
		0x81,0x81,0x00,0xef,0x88,0x5b,0x87,0xe2,0x5c,0x29,0xd1,0xee,0x43,0x1d,0x1f,0x4a,
		0xce,0x78,0x7c,0x4f,0x88,0x2e,0x92,0xd1,0x2c,0x4b,0x27,0x66,0xc8,0x4f,0x89,0x26,
		0x31,0x06,0xef,0x0b,0xe0,0xdd,0xc4,0xde,0x3b,0xf0,0x61,0xa2,0xba,0x46,0xff,0xe7,
		0xc1,0x32,0xae,0x67,0xd3,0x37,0xf0,0x4d,0xc6,0x6d,0xae,0xb9,0xde,0x55,0x37,0x91,
		0xf4,0x98,0x9f,0x50,0xb2,0x24,0xc9,0x81,0x81,0x2b,0x53,0x88,0xd1,0xc2,0xb5,0xd5,
		0x33,0x49,0xa6,0x13,0x93,0xbf,0x66,0x11,0x99,0x5b,0x69,0x88,0xf1,0x43,0xc2,0x78,
		0xcc,0xc2,0x60,0xcf,0xe8,0x2a,0x6a,0xc4,0xa6,0x40,0x98,0x07,0xea,0xb3,0x26,0x64,
		0xb7,0xee,0x3f,0x1c,0x41,0xdf,0xe5,0x67,0xe0,0x97,0xfd,0x7a,0xfe,0x85,0x20,0x87,
		0x1a,0x1c,0x53,0x02,0x81,0x81,0x00,0xdb,0xca,0x58,0x22,0x88,0xc8,0x1a,0x30,0xd6,
		0x58,0x1f,0x27,0x20,0x50,0xa5,0xfe,0xa7,0x73,0xd7,0x11,0x30,0x1b,0xdd,0x6e,0xca,
		0xc7,0x21,0x4e,0x00,0xc0,0xfc,0xb9,0x49,0x8f,0x1e,0x58,0x50,0x86,0xb1,0x44,0x63,
		0xcc,0x89,0xa0,0x2c,0x2e,0x77,0xa8,0x92,0x71,0xc0,0x58,0xc6,0xf6,0x9f,0x3c,0xd1,
		0x35,0x53,0x99,0x4d,0x8a,0x76,0xcf,0x62,0xad,0x9e,0x27,0x52,0x17,0x72,0x0d,0x62,
		0xeb,0x6e,0x88,0x8e,0xc4,0x50,0x9c,0xea,0x47,0x4b,0x53,0xf3,0x7b,0x45,0x8f,0xc9,
		0x56,0xdd,0x31,0xdf,0x3b,0xb6,0xec,0x7d,0xe6,0x59,0xe8,0x8e,0x7e,0x70,0x9f,0xe3,
		0xbe,0x6a,0x8d,0x37,0x26,0x4c,0x3d,0x20,0xe7,0xd0,0x88,0x27,0x6c,0xe7,0x65,0x1a,
		0x40,0x3c,0x68,0xd6,0xc4,0x7e,0xa5,0x02,0x81,0x81,0x00,0x9d,0x7e,0x87,0xc8,0x51,
		0xd2,0x8d,0x80,0xc5,0xeb,0x84,0xf3,0x75,0x49,0x4a,0xb9,0x59,0xc5,0xcd,0xf1,0xa4,
		0xed,0x3d,0xc0,0xfb,0x78,0xcb,0xaa,0xfe,0xdc,0x8f,0x95,0x8f,0xb6,0xdb,0xba,0x3c,
		0xf1,0x26,0x3d,0xdc,0x34,0x24,0xc8,0xd0,0x46,0x1c,0x9f,0xc6,0x0e,0x80,0x22,0x55,
		0xd1,0x19,0x7f,0x20,0x21,0x0a,0xe1,0x0d,0xeb,0xc8,0x8a,0x40,0x11,0xd1,0xd4,0x58,
		0x7d,0x68,0xd2,0x75,0x0c,0x8e,0xbc,0xe6,0x20,0xe1,0xd4,0xfb,0xdc,0x52,0xa5,0xb6,
		0xfe,0xa1,0xb7,0x43,0x5e,0x77,0x52,0x20,0x01,0x69,0xf1,0x23,0xe2,0xa0,0x39,0x31,
		0x71,0xaa,0xd9,0x0b,0xa3,0x8b,0x05,0xbc,0x85,0x9f,0x76,0x09,0x8b,0x5a,0xbe,0xc8,
		0xcd,0x48,0xc2,0x57,0x23,0x90,0xbe,0xf1,0x75,0xfe,0x97,0x02,0x81,0x80,0x0e,0xed,
		0x60,0x2d,0x00,0x43,0x2e,0xdc,0x30,0x42,0x8d,0xe3,0x17,0x63,0xc2,0xd2,0x57,0xc7,
		0x1b,0x4d,0x34,0x8a,0x3f,0xf0,0xbd,0x5e,0xe6,0xd9,0x28,0x5d,0xf3,0x0c,0x16,0x7d,
		0xdc,0xde,0xbc,0xa1,0x59,0x3a,0xbe,0xd8,0x6e,0x64,0x6d,0x7b,0xbe,0x6e,0xb9,0x7b,
		0x76,0x47,0xd1,0x43,0x80,0xaf,0x1d,0xba,0x54,0x72,0x2d,0xfd,0x00,0x72,0xb7,0x4d,
		0xf9,0x56,0xcc,0x51,0x81,0x52,0x7d,0x6c,0x65,0xf6,0x6a,0x53,0xd4,0xf1,0xc3,0x4b,
		0x92,0x47,0x22,0x5b,0x35,0xef,0x3d,0x0a,0x64,0x3d,0x75,0xde,0xd5,0x5e,0x9c,0x72,
		0x5f,0x9a,0xd6,0xca,0xa9,0x95,0x82,0x5f,0x35,0x57,0x5f,0xee,0x7e,0xf1,0x0b,0xe2,
		0x12,0x9c,0x9c,0xa8,0xdd,0xd2,0x55,0x05,0x15,0xd5,0x3c,0xf8,0xdd,0x6d,0x02,0x81,
		0x80,0x10,0x1d,0x1f,0x49,0xaf,0xb0,0x4d,0x06,0x5b,0xd8,0xc2,0x90,0x01,0xd2,0x12,
		0xe7,0x37,0xbb,0xa6,0x96,0x10,0x85,0x74,0xa3,0x30,0xa9,0xcb,0xac,0xc5,0x1b,0xf6,
		0xc9,0x65,0x94,0xf3,0x7f,0xba,0x9c,0x8d,0xe1,0x56,0xc2,0x26,0x37,0x1d,0x49,0x90,
		0x21,0x91,0xe1,0xf6,0x9d,0x84,0xa1,0x35,0x21,0x93,0xbf,0xf2,0x9e,0x31,0x8e,0xb3,
		0x6e,0xde,0x6f,0xfb,0xab,0x93,0xb6,0x55,0x5e,0xcf,0x6a,0xdd,0xca,0x9e,0x13,0x4c,
		0x20,0x22,0x0a,0x2d,0x24,0xb0,0x3a,0x6d,0x23,0xef,0x8c,0x60,0x8a,0x51,0x86,0x76,
		0x9f,0x66,0xfe,0x2f,0x6b,0x21,0xbc,0x39,0xf6,0x27,0x7a,0xd7,0x06,0x03,0x8b,0x90,
		0x78,0x72,0xc4,0x71,0x66,0x09,0xc2,0x23,0x76,0x21,0x30,0xef,0x03,0x61,0x64,0x82,
		0xf2);

/**
 * RSA OAEP SHA384 decryption test vectors from https://boringssl.googlesource.com/
 */
static rsa_oaep_test_t rsa_2048_oaep_sha384_tests[] = {
	{ chunk_from_chars(                                                          /* 2 */
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x00,0x51,0x19,0x0f,0x85,0xcf,0x3b,0xd0,0xe9,0xbd,0xf5,0xc9,0x37,0x59,0x3d,0x22,
	  	0x19,0xdb,0x50,0xba,0x7d,0x05,0x12,0xd3,0x88,0x07,0xb3,0x1e,0xaa,0x93,0x10,0xf0,
	  	0x12,0x03,0x20,0xf7,0xfa,0xd7,0xc6,0xb1,0x35,0xf2,0x9c,0xb1,0x7f,0xfa,0x5b,0x13,
	  	0x5c,0x0e,0xbb,0x1c,0x4c,0x81,0x12,0xa6,0x69,0x12,0xaa,0x92,0xbb,0xb3,0x19,0x1d,
	  	0x5a,0x28,0x48,0x84,0xf1,0x94,0x60,0xdf,0xb9,0xbf,0x23,0x2f,0x0d,0xb6,0xc2,0x07,
	  	0xff,0x21,0xd5,0x37,0x12,0xe1,0xc8,0xd9,0x0b,0xec,0x03,0x64,0x26,0xce,0xf5,0x6a,
	  	0x01,0x7e,0x61,0x75,0xbf,0x8b,0xf2,0x38,0xc3,0x20,0x50,0x18,0x88,0x69,0x30,0x1e,
	  	0xf7,0x8d,0x39,0x84,0x76,0x20,0x38,0xbb,0x72,0xa9,0x82,0xd5,0xc9,0xc9,0x50,0x18,
	  	0x20,0x19,0x6d,0x68,0xc7,0x98,0xea,0x29,0xd0,0xd6,0x0f,0x76,0x28,0x62,0x5f,0xfe,
	  	0x97,0x5e,0xe0,0x75,0x7c,0x6b,0x70,0xbf,0x87,0xb8,0x21,0x9d,0xdf,0x9e,0x6a,0xd3,
	  	0x24,0xec,0x97,0x51,0xfd,0x36,0x64,0x88,0x73,0x36,0xbc,0xaa,0x7a,0xf7,0x1c,0x93,
	  	0x08,0x3d,0xd9,0x67,0x02,0xe6,0x08,0x9e,0x2b,0x67,0xab,0xb7,0x63,0x19,0x3f,0x95,
	  	0xd4,0x22,0x2b,0x91,0x77,0xc1,0x7b,0x47,0x26,0x99,0x82,0x40,0xce,0x74,0x20,0x5f,
	  	0x18,0xcb,0x4b,0x17,0xbb,0xfc,0x73,0x18,0x51,0x76,0xad,0x76,0x1c,0x1d,0x48,0xe6,
	  	0x33,0xa4,0xac,0x7a,0xdc,0x2f,0x61,0xdc,0x95,0x79,0x8e,0x53,0x0d,0x20,0x81,0x7b,
	  	0x72,0x13,0x11,0xbe,0x82,0x4d,0xfc,0xf7,0x62,0x9f,0xbc,0x53,0x35,0x43,0x28,0x0c)
	},
	{ chunk_from_chars(                                                          /* 3 */
		0x54,0x65,0x73,0x74),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x0a,0x50,0x67,0xa9,0x7a,0x17,0x11,0x4b,0xe5,0xa7,0x16,0x97,0xf1,0x56,0x94,0xd9,
	  	0x8c,0xdd,0x6e,0x79,0xdc,0xba,0x17,0x26,0xa3,0x8c,0xa6,0x60,0xae,0x4f,0xd9,0xa1,
	  	0x33,0x6d,0xf2,0xde,0x79,0x71,0x8c,0xea,0x17,0x6a,0x36,0x5c,0x46,0xa2,0x4d,0x04,
	  	0x68,0xc2,0xa7,0x19,0xcc,0x18,0x35,0xe5,0xb6,0x0a,0xa6,0x44,0x24,0xa8,0xde,0xa0,
	  	0x92,0x1f,0xb6,0x41,0x45,0x8c,0x27,0x9c,0x74,0xa7,0xac,0x74,0x47,0x41,0x26,0xbe,
	  	0xcc,0x25,0xc8,0x4c,0xb8,0xe4,0x11,0xcc,0x7a,0x11,0x9c,0x78,0x29,0x8a,0x2f,0x45,
	  	0x65,0x8c,0x75,0x90,0xc3,0xb4,0x0d,0xbc,0xeb,0x7c,0x41,0xc6,0x7e,0x73,0xd4,0x1b,
	  	0x46,0xac,0x55,0xcb,0xea,0x54,0x1c,0x64,0xa9,0x1c,0x52,0x1e,0x37,0x78,0x90,0xcc,
	  	0x03,0x1a,0x49,0x33,0xa2,0x37,0x60,0x9d,0xce,0x1d,0x9f,0x76,0x68,0x46,0x41,0x41,
	  	0x7a,0x44,0x75,0x96,0x62,0xe1,0x66,0xd8,0x37,0xe7,0x7e,0x9b,0x8b,0x3f,0xd3,0x34,
	  	0x3c,0xad,0x26,0xde,0x86,0xdb,0x4f,0x91,0x4a,0xe8,0x38,0x93,0xfe,0x96,0x58,0x65,
	  	0x4d,0x09,0x32,0xb6,0xa9,0x8c,0xa5,0x73,0x56,0x69,0xd6,0x67,0x18,0x6d,0x89,0x4b,
	  	0x5d,0x47,0x2a,0x1e,0xfe,0xf4,0xb2,0xe0,0x60,0x35,0x18,0x9f,0xea,0xf9,0x60,0xca,
	  	0x12,0x42,0xb6,0x20,0x26,0x4e,0x71,0xc4,0x73,0x07,0xa8,0xa3,0x8b,0xba,0xe9,0xbc,
	  	0x3d,0x1b,0xd5,0x50,0x8a,0xc3,0x7f,0x1b,0xa7,0x70,0xbf,0xc3,0x6a,0x50,0xa5,0x43,
	  	0xee,0x20,0x4e,0x5e,0xff,0x53,0x10,0xfe,0x3b,0xf4,0x30,0xdc,0x69,0x08,0xeb,0x9e)
	},
	{ chunk_from_chars(                                                          /* 4 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x76,0x7b,0xcf,0x45,0x0f,0x41,0x6e,0x42,0x89,0x80,0xa4,0x88,0xbc,0x83,0x07,0x67,
	  	0x49,0x76,0x25,0x30,0x08,0xd8,0x13,0x40,0xa2,0xf5,0x19,0xa1,0xf4,0xb9,0x36,0xab,
	  	0xf0,0x20,0x28,0xc2,0xe7,0x61,0x4f,0x4a,0xa3,0x1a,0x8b,0x3e,0x17,0xc9,0x6b,0x49,
	  	0x22,0x34,0x0a,0x6a,0x96,0x96,0xfd,0x4f,0xf3,0x58,0xc1,0xa8,0x4c,0x34,0x7b,0x40,
	  	0x50,0x33,0x45,0x9f,0x91,0x92,0xe9,0xf7,0xd6,0x11,0xca,0x2d,0xa6,0xa6,0x69,0xa8,
	  	0x95,0xd4,0x91,0xa3,0xc1,0x90,0x8c,0xb5,0x98,0x73,0x0b,0x59,0x6d,0x66,0xb1,0xd4,
	  	0xb7,0x61,0x7f,0x33,0x76,0x6a,0xf3,0x85,0xc3,0x33,0xf0,0x37,0xe7,0x34,0x1c,0x96,
	  	0x80,0xf9,0x85,0x4f,0x6a,0xb4,0xf2,0xc9,0x86,0x46,0x2d,0x87,0x4f,0x4d,0xc6,0x6c,
	  	0x6f,0xa4,0xf4,0x9f,0xd3,0xef,0x33,0xd7,0xc8,0xcd,0x81,0xd7,0x60,0x84,0x11,0xaa,
	  	0x61,0x0e,0x5e,0x91,0x01,0xdb,0xdb,0xa5,0x22,0xf7,0x5c,0xd1,0x2c,0x57,0x5b,0x42,
	  	0xd0,0x82,0x81,0x12,0x27,0xba,0x10,0x31,0x9f,0x6e,0xa0,0xc9,0xa2,0xeb,0x6a,0x3e,
	  	0x63,0xcf,0xe0,0xc1,0xba,0xb1,0x8f,0x29,0x09,0x1f,0xc5,0xd6,0x37,0x27,0x85,0x20,
	  	0x8a,0x83,0x0f,0x73,0x2f,0x8a,0x06,0x35,0xf5,0xee,0x7b,0x0b,0x9d,0x2d,0x62,0xff,
	  	0x9f,0x1a,0x6c,0xb8,0x8d,0xa2,0x53,0xad,0x5f,0x0d,0x56,0xe5,0x72,0x54,0x8a,0x35,
	  	0x0a,0x7e,0x12,0xff,0x84,0x6d,0x09,0xad,0x53,0x52,0xfa,0x45,0xde,0x48,0x43,0x86,
	  	0xb5,0x29,0xe6,0xf1,0x4a,0x1e,0x80,0x55,0x14,0x53,0x94,0xf8,0xd4,0xf9,0x1b,0x80)
	},
	{ chunk_from_chars(                                                          /* 8 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
	  	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	  chunk_from_chars(
	  	0xa5,0x9e,0xd1,0x88,0x17,0x82,0xb3,0xa1,0x51,0x25,0x4f,0xd0,0x10,0x2d,0x0c,0x78,
	  	0x26,0x9c,0xe0,0x15,0xa5,0x42,0x76,0x77,0x63,0xd7,0x86,0xbb,0xb8,0x8d,0x7c,0x76,
	  	0x4e,0x6a,0x24,0x49,0xf4,0x9b,0xb8,0x31,0x34,0x30,0xed,0x48,0x0c,0xe6,0xd6,0x55,
	  	0x83,0x50,0x2f,0x64,0x7f,0x43,0x9f,0xfb,0x57,0x8c,0xba,0x62,0x67,0xdd,0xb4,0x92,
	  	0xdc,0x74,0xff,0xab,0x44,0xce,0xbb,0xc1,0x1d,0xf4,0x4f,0xa6,0xe8,0x38,0x99,0x71,
	  	0x9c,0x74,0x32,0xe5,0x09,0xae,0x3a,0x58,0x71,0x7f,0x45,0x2b,0xdb,0x53,0x1f,0x01,
	  	0xb7,0x0f,0x95,0x41,0xca,0xdb,0x50,0x4b,0x94,0x2f,0x82,0xb1,0x03,0x2a,0x5d,0xaa,
	  	0x27,0x17,0x9d,0x83,0xdc,0x63,0xe0,0xf3,0x42,0xd5,0xf7,0x69,0xfa,0xfc,0x07,0x50,
	  	0x68,0x74,0x70,0x4f,0xab,0x17,0xd6,0x1c,0xbd,0xa6,0x98,0xa2,0x01,0xfa,0x42,0x73,
	  	0x8f,0xb2,0x1e,0x9e,0xf8,0x82,0x8c,0xfa,0x0e,0x0b,0x4a,0x5e,0xe8,0x6e,0x63,0x9d,
	  	0x69,0x64,0x27,0x48,0xcc,0x76,0x4a,0x4f,0xa2,0x7f,0xdb,0x18,0xc4,0x22,0xf1,0x25,
	  	0x7b,0x9a,0x7c,0xaa,0xf2,0x43,0x0e,0x6a,0x34,0x7b,0xba,0xa1,0x70,0xb7,0x7f,0xc7,
	  	0xf2,0xaa,0xb7,0xad,0x26,0x73,0x32,0x4c,0x17,0xac,0xff,0x7f,0x67,0xcc,0x00,0x52,
	  	0x54,0x83,0xc5,0xa3,0x90,0xc0,0x84,0xec,0xeb,0x0d,0xde,0xa6,0x37,0xf0,0x9c,0xc1,
	  	0x15,0x86,0x38,0xee,0xde,0xad,0x7b,0x38,0xa1,0x4e,0x99,0x39,0xc5,0x55,0xcf,0xb1,
	  	0x88,0x1d,0xae,0x08,0x47,0x9d,0x83,0x8c,0x1d,0xc1,0xaf,0x3b,0x81,0x34,0x83,0x05)
	},
	{ chunk_from_chars(                                                          /* 9 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13),
	  chunk_from_chars(
		0x54,0x90,0xda,0x68,0x47,0x78,0xd7,0x36,0x66,0xab,0x3c,0xdd,0xb1,0x45,0xbf,0x50,
		0xc9,0x14,0xed,0x7f,0xc5,0xcc,0xca,0x7f,0xa5,0xae,0xc0,0x7e,0x45,0xd9,0x25,0xfd,
		0x40,0x4e,0x34,0xc8,0x99,0xb3,0x62,0x0e,0xe0,0x10,0xcb,0xdf,0x02,0xf1,0x90,0xbf,
		0x23,0xf4,0x7c,0x07,0xf9,0x95,0x14,0x2b,0xfa,0x0b,0x1b,0xdd,0x98,0x88,0x72,0x67,
		0x1a,0xa4,0xcc,0xab,0xa8,0xfc,0xc6,0x68,0xcf,0x41,0x3a,0x62,0x7b,0xdf,0x27,0xf8,
		0xb6,0x80,0xf1,0xb9,0xd6,0x33,0xcf,0x3e,0x2a,0x3c,0x0b,0x83,0xa9,0xeb,0xec,0x59,
		0x1c,0x9f,0xd9,0xf1,0xa4,0xbc,0x99,0x7a,0xd8,0x36,0xbf,0x5d,0xb9,0xa8,0x01,0x5c,
		0x34,0xbe,0xf1,0x13,0x92,0xef,0x8b,0xfa,0x4b,0x27,0x8b,0x9b,0xa8,0xf3,0x15,0xe2,
		0x56,0x90,0x82,0x99,0x81,0x22,0x45,0xdc,0x53,0x4d,0xea,0xcb,0x9b,0x97,0xb7,0x64,
		0x51,0x62,0xab,0x66,0x97,0x21,0x26,0x81,0x77,0x40,0x47,0x99,0x17,0x3f,0x4f,0x9a,
		0x1e,0x90,0xab,0xaf,0x04,0xcf,0x78,0xa1,0xea,0x8a,0x95,0xe8,0x6c,0x34,0x35,0x53,
		0x15,0x35,0x62,0xe3,0x17,0x5e,0xa1,0x35,0x99,0x8e,0x20,0xcb,0x51,0x38,0x6e,0x93,
		0x4e,0xe2,0x86,0x76,0xe1,0x0a,0x5a,0x73,0xb5,0x9f,0xa1,0xca,0x3b,0x5c,0x96,0xae,
		0x43,0x32,0x38,0x83,0xab,0x5a,0x9a,0x92,0x14,0xba,0xa4,0x61,0x04,0x6e,0x3f,0x60,
		0x0e,0xac,0xed,0x75,0xcc,0xfb,0x88,0x7f,0xdb,0x11,0xd5,0xd2,0x02,0x34,0x79,0x18,
		0xcb,0x42,0x27,0x63,0x70,0xef,0x6f,0xb4,0x23,0x30,0xd3,0x92,0x27,0x71,0xe7,0xb5)
	},
	{ chunk_from_chars(                                                         /* 10 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f),
	  chunk_from_chars(
		0xaa,0xec,0x36,0x6e,0x45,0xe9,0xcb,0xee,0x80,0x4f,0x4b,0xea,0xb0,0xc7,0xd4,0xa8,
		0x02,0x73,0x1a,0xe1,0xe2,0x2d,0x93,0xe3,0xdc,0x0f,0xa4,0x86,0x8d,0xd8,0x49,0xc0,
		0xcd,0xfc,0x52,0x06,0x60,0x51,0xcf,0x12,0x88,0x7c,0xcc,0x09,0x25,0x73,0x01,0x02,
		0x6b,0xe0,0xe4,0xeb,0xe2,0xd6,0x0d,0xbe,0xc6,0xde,0x54,0x7d,0x1d,0xd4,0xf7,0x78,
		0x68,0x2f,0xa9,0xd8,0x8d,0xcc,0xb0,0xea,0xb5,0xe6,0xcb,0x07,0xfe,0x7c,0x3a,0x36,
		0x64,0xb5,0xf3,0x1b,0xd6,0x8c,0xdb,0xe9,0xb8,0x8f,0xec,0x68,0x4b,0x9e,0x4a,0x02,
		0x3f,0x0e,0x78,0x2e,0x5e,0x9a,0x32,0x86,0x28,0xdb,0x6c,0x3d,0x94,0xa1,0x08,0x2b,
		0xd0,0xd3,0xe4,0x53,0xce,0x26,0x28,0xc1,0xf1,0x7a,0x76,0x7c,0x01,0x7d,0x72,0x50,
		0x84,0xdd,0x0b,0x7d,0x71,0x52,0x41,0xb8,0x08,0xb2,0x53,0x74,0x64,0xdc,0xc6,0xee,
		0xb5,0xaf,0x4c,0x4c,0x18,0x70,0x2b,0xcd,0xc7,0x33,0xc5,0x3f,0x35,0xd8,0x05,0xf8,
		0x13,0xe7,0x72,0xac,0x32,0xbc,0xaf,0xfe,0x02,0x5e,0xf0,0xfa,0x89,0x31,0x84,0x9f,
		0xeb,0xf9,0x9c,0x9e,0x73,0xcf,0x5e,0x47,0x30,0xa1,0x49,0x24,0xd9,0xf0,0x74,0x06,
		0x3b,0x4d,0x80,0xfa,0xf3,0xe8,0xbc,0xe1,0x93,0x0d,0xaa,0x24,0x1e,0xce,0xe3,0xb6,
		0x84,0xd6,0x67,0x4d,0xc4,0x1b,0x8c,0xe5,0x3a,0xa8,0xc0,0x00,0x2b,0x02,0x00,0xfa,
		0x39,0x37,0x20,0xed,0x96,0x78,0x22,0xeb,0x2e,0xdd,0x74,0xa1,0x95,0x42,0x3e,0xba,
		0x5d,0xd6,0xfe,0x39,0x15,0x0d,0xcb,0x69,0x7c,0xc1,0xf3,0x55,0x3f,0x84,0xb9,0xfe)
	},
	{ chunk_from_chars(                                                         /* 11 */
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x34,0x8f,0x66,0x80,0xd2,0x5d,0x93,0xe7,0xf8,0xf1,0x7f,0x24,0x10,0xdd,0x4b,0xdc,
	  	0x8a,0x2b,0xa5,0xc9,0xd2,0x9b,0x17,0xf3,0xdc,0x1e,0x30,0x8d,0x86,0x8e,0xf3,0xe1,
	  	0x12,0x19,0x82,0x63,0x5c,0x60,0xfd,0x38,0x6f,0x77,0x56,0x63,0x70,0xf5,0xbc,0x0d,
	  	0x34,0xe9,0x59,0x66,0x37,0xc5,0x9b,0x00,0xa6,0xeb,0x66,0x5b,0xf5,0x70,0x71,0x26,
	  	0x04,0x83,0xf7,0x2a,0x5c,0xf0,0xbd,0x6a,0xd8,0x06,0xa0,0x49,0xaa,0x67,0x0f,0xfa,
	  	0xc2,0x06,0x67,0x2f,0xad,0x6e,0x20,0x6d,0x9e,0x13,0xf8,0x2d,0x89,0xcd,0x36,0x75,
	  	0xa8,0x88,0x87,0x3c,0xfd,0xd1,0xea,0x20,0xfe,0xec,0x04,0xac,0x9e,0x1d,0x4d,0xae,
	  	0xbc,0x58,0xae,0x9e,0xbb,0x84,0x45,0x9a,0x34,0x81,0xd2,0x68,0x46,0x8f,0x6e,0x97,
	  	0xbb,0xad,0x00,0x11,0x24,0x75,0x76,0xcd,0xce,0xa3,0x19,0xb6,0x8b,0x19,0xce,0x7f,
	  	0x25,0x73,0xcb,0x3d,0x71,0x1e,0x07,0x8b,0x9e,0x20,0x6f,0xea,0x72,0x9e,0x5b,0x61,
	  	0xd6,0x7f,0x40,0x0a,0x33,0x5a,0xed,0xb4,0x8f,0x67,0x3a,0xf1,0xa8,0x1c,0x29,0x98,
	  	0xd8,0x0b,0xa3,0x4a,0xc1,0x85,0x30,0xcc,0xef,0x71,0x83,0x86,0x58,0xe8,0x93,0x7c,
	  	0x12,0x7f,0x5d,0x5c,0xe9,0x3a,0x52,0x5a,0x2e,0xc6,0x0c,0x98,0x56,0x9d,0x78,0x55,
	  	0x92,0x49,0x0f,0xcf,0x3c,0x72,0xd5,0x37,0x91,0xf8,0xe8,0x12,0x44,0x29,0x2f,0x18,
	  	0x24,0x26,0x45,0x18,0xbb,0x82,0x7a,0xdf,0x12,0x05,0xe6,0xa2,0xe1,0xca,0x7d,0x01,
	  	0x19,0x5d,0xd4,0xc3,0x61,0x6a,0xcb,0x75,0x80,0xc0,0xc9,0xc5,0x72,0x02,0x4a,0x4f)
	}
};

static void test_rsa_oaep(chunk_t key, encryption_scheme_t encryption_scheme,
						  rsa_oaep_test_t *test)
{
	private_key_t *privkey;
	public_key_t *pubkey;
	chunk_t msg = chunk_empty, ct = chunk_empty;
	bool success;

	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_ANY,
								 BUILD_BLOB_ASN1_DER, key, BUILD_END);
	ck_assert(privkey != NULL);
	ck_assert(privkey->get_type(privkey) == KEY_RSA);

	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);

	/* decrypt */
	success = privkey->decrypt(privkey, encryption_scheme, &test->label,
							   test->ct, &msg);
	if (test->label.len > 0 && !success)
	{
		warn("RSA OAEP labels might not be supported");
		goto end;
	}
	ck_assert(success);
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&msg);

	/* encrypt and decrypt again */
	ck_assert(pubkey->encrypt(pubkey, encryption_scheme, &test->label,
							  test->msg, &ct));
	ck_assert(privkey->decrypt(privkey, encryption_scheme, &test->label,
							   ct, &msg));
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&ct);
	chunk_free(&msg);

end:
	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}

START_TEST(test_rsa_2048_oaep_sha384)
{
	test_rsa_oaep(rsa_2048_key, ENCRYPT_RSA_OAEP_SHA384,
				 &rsa_2048_oaep_sha384_tests[_i]);
}
END_TEST

Suite *rsa_oaep_sha384_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("rsa_oaep_sha384");

	tc = tcase_create("rsa_2048_oaep_sha384");
	tcase_add_loop_test(tc, test_rsa_2048_oaep_sha384, 0, countof(rsa_2048_oaep_sha384_tests));
	suite_add_tcase(s, tc);

	return s;
}
