#!/bin/bash
# The script requires root permissions

set -e

function print_help() {
    cat <<END
This is a simplified trento-agent installation script.
The Trento Agent is a single process that discovers information of your target SAP infrastructure and pushes them to the control plane, Trento Web.

Usage:

  sudo ./install-agent.sh --server-url <trento-server-url> --api-key <your-api-key> --facts-service-url <amqp-service-url>

Arguments:
  --server-url        The trento server url.
  --facts-service-url The fact gathering service url.
  --api-key           The API key generated byt the trento server installation.
  --rolling           Use the rolling version instead of the stable one.
  --interval          The polling interval in seconds for the discoveries.
  --help              Print this help.
END
}

case "$1" in
--help)
    print_help
    exit 0
    ;;
esac

if [ "$EUID" -ne 0 ]; then
    echo "Please run as root."
    exit
fi

ARGUMENT_LIST=(
    "server-url:"
    "facts-service-url"
    "api-key:"
    "rolling"
    "interval:"
)

opts=$(
    getopt \
        --longoptions "$(printf "%s," "${ARGUMENT_LIST[@]}")" \
        --name "$(basename "$0")" \
        --options "" \
        -- "$@"
)

eval set "--$opts"

while [[ $# -gt 0 ]]; do
    case "$1" in
    --server-url)
        SERVER_URL=$2
        shift 2
        ;;

    --facts-service-url)
        FACTS_SERVICE_URL=$2
        shift 2
        ;;

    --api-key)
        API_KEY=$2
        shift 2
        ;;

    --rolling)
        USE_ROLLING=true
        shift 1
        ;;

    --interval)
        INTERVAL=$2
        shift 2
        ;;

    *)
        break
        ;;
    esac
done

AGENT_CONFIG_PATH="/etc/trento"
AGENT_CONFIG_FILE="$AGENT_CONFIG_PATH/agent.yaml"
AGENT_CONFIG_TEMPLATE='
server-url: @SERVER_URL@
facts-service-url: @FACTS_SERVICE_URL@
api-key: @API_KEY@
cloud-discovery-period: @INTERVAL@s
cluster-discovery-period: @INTERVAL@s
host-discovery-period: @INTERVAL@s
sapsystem-discovery-period: @INTERVAL@s
'

. /etc/os-release
if [[ ! $PRETTY_NAME =~ "SUSE" ]]; then
    echo "Trento does not support non-SUSE operating systems. Exiting."
    exit 1
fi

echo "Installing trento-agent..."

function gather_configuration() {
    if [[ -z "$SERVER_URL" ]]; then
        read -rp "Please provide the server url: " SERVER_URL </dev/tty
    fi
    if [[ -z "$FACTS_SERVICE_URL" ]]; then
        read -rp "Please provide the facts service url: " FACTS_SERVICE_URL </dev/tty
    fi
    if [[ -z "$API_KEY" ]]; then
        read -rp "Please provide the API key: " API_KEY </dev/tty
    fi
}

function install_rpm() {
    if [[ -n "$USE_ROLLING" ]]; then
        TRENTO_REPO=${TRENTO_REPO:-"https://download.opensuse.org/repositories/devel:/sap:/trento:/factory/standard/devel:sap:trento:factory.repo"}
        TRENTO_REPO_KEY=${TRENTO_REPO_KEY:-"https://download.opensuse.org/repositories/devel:/sap:/trento:/factory/standard/repodata/repomd.xml.key"}
    else
        TRENTO_REPO=${TRENTO_REPO:-"https://download.opensuse.org/repositories/devel:/sap:/trento/standard/devel:sap:trento.repo"}
        TRENTO_REPO_KEY=${TRENTO_REPO_KEY:-"https://download.opensuse.org/repositories/devel:/sap:/trento/standard/repodata/repomd.xml.key"}
    fi

    rpm --import "${TRENTO_REPO_KEY}" >/dev/null
    path=${TRENTO_REPO%/*}/
    if zypper lr --details | cut -d'|' -f9 | grep "$path" >/dev/null 2>&1; then
        echo "* $path repository already exists. Skipping."
    else
        echo "* Adding Trento repository: $path."
        zypper ar "$TRENTO_REPO" >/dev/null
    fi
    zypper ref >/dev/null
    if which trento >/dev/null 2>&1; then
        echo "* Trento Agent is already installed. Updating trento"
        zypper up -y trento-agent >/dev/null
    else
        echo "* Installing trento"
        zypper in -y trento-agent >/dev/null
    fi
}

function setup() {
    local interval=${INTERVAL:-"10"}

    echo "* Generating trento-agent config..."

    mkdir -p ${AGENT_CONFIG_PATH} && touch ${AGENT_CONFIG_FILE}

    echo "$AGENT_CONFIG_TEMPLATE" |
        sed "s|@SERVER_URL@|${SERVER_URL}|g" |
        sed "s|@FACTS_SERVICE_URL@|${FACTS_SERVICE_URL}|g" |
        sed "s|@API_KEY@|${API_KEY}|g" |
        sed "s|@INTERVAL@|${interval}|g" \
            >${AGENT_CONFIG_FILE}
}

function start_service() {
    if systemctl is-active --quiet trento-agent; then
        echo "* Warning: trento-agent systemd unit was already running. Restarting..."
        systemctl restart trento-agent
    else
        echo -e "Now you can start trento-agent with: \033[1msystemctl start trento-agent\033[0m"
        echo -e "Please make sure the \033[1mserver\033[0m is running before starting the agent."
    fi
}

gather_configuration
install_rpm
setup
start_service
