#! /usr/bin/perl -w

use strict;
use Data::Dumper;
use Date::Calc qw(Mktime);
use Getopt::Long;

my @heimdal = ();
my @mit = ();
my $debug = 0;
my $infile = "";
my $outfile = "";
my $stashfile = "";
my $help = "";
my $encProg = "/usr/lib/mit/sbin/EncryptWithMasterKey";
my $encType = "des3-cbc-sha1";

sub flags_to_attr {
    my $flags = shift;
    my $attr = 0;

    $attr |= 1<<2 if(!!($flags & 0x00000001));   # initial
    $attr |= 1<<1 if(!($flags & 0x00000002));   # forwardable
    $attr |= 1<<4 if(!($flags & 0x00000004));   # proxiable
    $attr |= 1<<3 if(!($flags & 0x00000008));   # renewable
    $attr |= 1<<0 if(!($flags & 0x00000010));   # postdate
    $attr |= 1<<12 if(!($flags & 0x00000020));   # server
    #$attr |= 1<<6 if(!!($flags & 0x00000040));   # client  ???

    $attr |= 1<<6 if(!!($flags & 0x00000080));   # invalid
    $attr |= 1<<7 if(!!($flags & 0x00000100));   # require preauth
    $attr |= 1<<13 if(!!($flags & 0x00000200));   # change_pw

    print "FLAGS:$flags ATTR:$attr\n" if($debug);

    return $attr;
}

sub time2sec {
    my $date = shift;

    if($date eq "-") {
        return 0;
    }

    $date =~ /^(\d\d\d\d)(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)$/;
    if(!defined $1 || !defined $2 || !defined $3 ||
       !defined $4 || !defined $5 || !defined $6 ) {
        print STDERR "Can not convert date($date)\n";
        exit 1;
    }
    my ($year, $mon, $day, $hour, $min, $sec) = ($1, $2, $3, $4, $5, $6);

    my $time = Mktime($year, $mon, $day, $hour, $min, $sec);

    return $time;
}

sub gen_tl_data {
    my $date = shift;      # e.g. 20050112120600
    my $princ = shift;

    my $time = time2sec($date);

    my $hextime = sprintf("%08x", $time);
    print "HEXTIME: $hextime\n" if($debug);
    $hextime =~ /^([\da-fA-F]{2})([\da-fA-F]{2})([\da-fA-F]{2})([\da-fA-F]{2})$/ ;
    if(!defined $1 || !defined $2 || !defined $3 || !defined $4 ) {
        print STDERR "Can not convert date($date)\n";
        exit 1;
    }

    my $data = "$4$3$2$1".unpack("H*", "$princ")."00";
    print "Date: $date  Princ: $princ ==>> Data: $data\n" if($debug);
    return $data;
}

sub usage {
    print "usage: heimdal2mit-DumpConvert.pl --infile </path/to/heimdal.dump> \n";
    print "                                  --outfile </path/to/mit.dump> \n";
    print "                                  --stashfile </path/to/MIT/stashfile> \n";
    print "                                  [<other options>]\n";
    print " other options are:\n";
    print "       --encProg -ep       </path/to/EncryptWithMasterKey> (default is /usr/lib/mit/sbin/EncryptWithMasterKey)\n";
    print "       --encType -ec       <encryption type> (default is des3-cbc-sha1)\n";
    print "       --help -h           print help\n";
    print "       --debug -d          enable debug\n\n";
    print "heimdal2mit-DumpConvert.pl converts a decrypted heimdal kerberos dump file into MIT kerberos\n";
    print "dump format.\n"; 
    exit 1;
}


my $result = GetOptions ("infile|i=s"     => \$infile,
                         "outfile|o=s"    => \$outfile,
                         "stashfile|sf=s" => \$stashfile,
                         "encProg|ep=s"   => \$encProg,
                         "encType|ec=s"   => \$encType,
                         "help|h"         => \$help,
                         "debug|d"        => \$debug
                        );

if($help ) { 
    usage();
}

if(! defined $infile    || $infile eq ""    || ! -e $infile) {
    print "invalid infile: $infile\n\n";
    usage();
}

if(! defined $outfile   || $outfile eq "" ) {
    print "Missing outfile\n\n";
    usage();
}

if(! defined $stashfile || $stashfile eq "" || ! -e $stashfile) {
    print "invalid stashfile: $stashfile\n\n";
    usage();
}

if(! -e $encProg) {
    print "Unable to find EncryptWithMasterKey program. Please use --encProg option\n\n";
    usage();
}

if(!defined $encType || $encType eq "") {
    $encType = "des3-cbc-sha1";
}


open(HEIMDAL, "< $infile") or die "Can not open heimdal dump:$!";
@heimdal = <HEIMDAL>;
close HEIMDAL;

foreach my $line (@heimdal) {

    my @data = split(/\s/, $line);
    
    if($debug) {
        foreach (@data) {
            print $_."\n";
        }
        print "\n";
    }

    my $mitset = {};
    for(my $i=0; $i < @data; $i++) {
        if($i == 0) {  # principal
            $mitset->{"princ"} = $data[$i];
        } elsif($i == 1) {  # keys
            my @keyset = split(/:/, $data[$i]);
            $mitset->{"kvno"} = $keyset[0];
            my $j = 0;
            my $keycount = -1;
            my @keys = ();
            foreach my $dummy (@keyset) {
                if($j == 0) {
                    $j++;
                    next;
                }
 
                if(($j%4) == 1) {        # mkvno
                    $keycount++;
                    $keys[$keycount]->{"mkvno"} = $dummy;
                } elsif(($j%4) == 2) {   # enctype
                    $keys[$keycount]->{"enctype"} = $dummy;

                } elsif(($j%4) == 3) {   # keyvalue
                    # encrypt keyvalue with mit stash file
                    my $cmd = "$encProg -d $dummy -sf $stashfile -e $encType";
                    print STDERR "CMD: $cmd\n" if($debug);

                    my $ekey = `$cmd`;
                    chomp($ekey);

                    print STDERR "EKEY: $ekey\n" if($debug);
                    if(! defined $ekey || $ekey eq "") {
                        $ekey = $dummy;
                    }
                    $keys[$keycount]->{"keyvalue"} = $ekey;

                } elsif(($j%4) == 0) {   # salt (- means use normal salt)
                    $keys[$keycount]->{"salt"} = $dummy;

                } else {
                    print STDERR "Somthing very strage happens\n";
                }
                $j++;

            }
            $mitset->{"keys"} = \@keys;
        }elsif($i == 2) {  # creation date and principal
            my @ar = split(/:/, $data[$i]);
            $mitset->{"creatDate"} = $ar[0];
            $mitset->{"creatPrinc"} = $ar[1];
        } elsif($i == 3) {  # modification date and principal
            my @ar = split(/:/, $data[$i]);
            $mitset->{"modifDate"} = $ar[0];
            $mitset->{"modifPrinc"} = $ar[1];
        } elsif($i == 4) {  # not used
            #skip
        } elsif($i == 5) {  # expire date
            $mitset->{expire} = time2sec($data[$i]);
        } elsif($i == 6) {  # pw expire date
            $mitset->{pwexpire} = time2sec($data[$i]);
        } elsif($i == 7) {  # max ticket life
            $mitset->{"maxlife"} = $data[$i];
        } elsif($i == 8) {  # max renewable life
            $mitset->{"maxrenew"} = $data[$i];
        } elsif($i == 9) {  # flags
            $mitset->{"flags"} = $data[$i];
        } elsif($i == 10) {  # generation number
            $mitset->{"gennum"} = $data[$i];
        } else {
            die ("one field too much");
        }
    }
    push @mit, $mitset;
}

print Data::Dumper->Dump([@mit])."\n" if($debug);

my @mitout = ();

foreach my $dataset (@mit) {
    my $keynum = @{$dataset->{"keys"}};
    my $data = "";

    #
    # special heimdal/kerberos principals do not convert these
    # we want to import with -update option
    #
    next if($dataset->{'princ'} =~ /^default@/);
    next if($dataset->{'princ'} =~ /^kadmin\//);
    next if($dataset->{'princ'} =~ /^krbtgt\//);
    next if($dataset->{'princ'} =~ /^changepw\/kerberos@/);

    my $line = "princ\t38\t";
    $line .= length($dataset->{'princ'})."\t";
    #$line .= "num-of-tl_data"."\t";
    $line .= "1"."\t";
    #$line .= "num-of-keys"."\t";
    $line .= $keynum."\t";
    $line .= "0"."\t";                          # extra length
    $line .= $dataset->{"princ"}."\t";
    $line .= flags_to_attr($dataset->{"flags"})."\t";
    $line .= $dataset->{"maxlife"}."\t";
    $line .= $dataset->{"maxrenew"}."\t";
    $line .= $dataset->{"expire"}."\t";         # expiration
    $line .= $dataset->{"pwexpire"}."\t";       # passwd exp
    $line .= "0"."\t";                          # last succ auth
    $line .= "0"."\t";                          # last fail auth
    $line .= "0"."\t";                          # faild auth count

    #################################################################

    $line .= "2"."\t";   # type

    if(! defined $dataset->{"modifPrinc"}) {
        $data = gen_tl_data($dataset->{"creatDate"}, $dataset->{"creatPrinc"});
    } else {
        $data = gen_tl_data($dataset->{"modifDate"}, $dataset->{"modifPrinc"});
    }

    $line .= int(length($data)/2)."\t";                    # length data
    $line .= "$data"."\t";                                 # data

    foreach my $h_key (@{$dataset->{"keys"}}) {

        $line .= "1"."\t";                                 # key version
        $line .= $dataset->{"kvno"}."\t";                  # kvno
        $line .= $h_key->{"enctype"}."\t";                 # enctype / keytype
        $line .= int(length($h_key->{"keyvalue"})/2)."\t"; # key length
        $line .= $h_key->{"keyvalue"}."\t";                # key
    }
    push @mitout, $line."-1;\n"
}

open(MIT, "> $outfile") or die "Can not open file:$!";
print MIT "kdb5_util load_dump version 5\n";
print MIT @mitout;
close MIT;
